/*
 * Decompiled with CFR 0.152.
 */
package io.atomix.protocols.backup.session;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import io.atomix.cluster.ClusterMembershipEvent;
import io.atomix.cluster.ClusterMembershipEventListener;
import io.atomix.cluster.ClusterMembershipService;
import io.atomix.cluster.Member;
import io.atomix.primitive.Consistency;
import io.atomix.primitive.PrimitiveException;
import io.atomix.primitive.PrimitiveState;
import io.atomix.primitive.PrimitiveType;
import io.atomix.primitive.Recovery;
import io.atomix.primitive.Replication;
import io.atomix.primitive.event.EventType;
import io.atomix.primitive.event.PrimitiveEvent;
import io.atomix.primitive.operation.PrimitiveOperation;
import io.atomix.primitive.partition.PartitionId;
import io.atomix.primitive.partition.PrimaryElection;
import io.atomix.primitive.partition.PrimaryElectionEventListener;
import io.atomix.primitive.partition.PrimaryTerm;
import io.atomix.primitive.session.SessionClient;
import io.atomix.primitive.session.SessionId;
import io.atomix.protocols.backup.protocol.CloseRequest;
import io.atomix.protocols.backup.protocol.ExecuteRequest;
import io.atomix.protocols.backup.protocol.PrimaryBackupClientProtocol;
import io.atomix.protocols.backup.protocol.PrimaryBackupResponse;
import io.atomix.protocols.backup.protocol.PrimitiveDescriptor;
import io.atomix.utils.concurrent.ComposableFuture;
import io.atomix.utils.concurrent.ThreadContext;
import io.atomix.utils.event.EventListener;
import io.atomix.utils.logging.ContextualLoggerFactory;
import io.atomix.utils.logging.LoggerContext;
import java.net.ConnectException;
import java.time.Duration;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import org.slf4j.Logger;

public class PrimaryBackupSessionClient
implements SessionClient {
    private static final int MAX_ATTEMPTS = 50;
    private static final int RETRY_DELAY = 100;
    private Logger log;
    private final PrimitiveType primitiveType;
    private final PrimitiveDescriptor descriptor;
    private final ClusterMembershipService clusterMembershipService;
    private final PrimaryBackupClientProtocol protocol;
    private final PartitionId partitionId;
    private final SessionId sessionId;
    private final PrimaryElection primaryElection;
    private final ThreadContext threadContext;
    private final Set<Consumer<PrimitiveState>> stateChangeListeners = Sets.newIdentityHashSet();
    private final Map<EventType, Set<Consumer<PrimitiveEvent>>> eventListeners = Maps.newHashMap();
    private final PrimaryElectionEventListener primaryElectionListener = event -> this.changeReplicas(event.term());
    private final ClusterMembershipEventListener membershipEventListener = this::handleClusterEvent;
    private PrimaryTerm term;
    private volatile PrimitiveState state = PrimitiveState.CLOSED;

    public PrimaryBackupSessionClient(String clientName, PartitionId partitionId, SessionId sessionId, PrimitiveType primitiveType, PrimitiveDescriptor descriptor, ClusterMembershipService clusterMembershipService, PrimaryBackupClientProtocol protocol, PrimaryElection primaryElection, ThreadContext threadContext) {
        this.partitionId = (PartitionId)Preconditions.checkNotNull((Object)partitionId);
        this.sessionId = (SessionId)Preconditions.checkNotNull((Object)sessionId);
        this.primitiveType = primitiveType;
        this.descriptor = descriptor;
        this.clusterMembershipService = clusterMembershipService;
        this.protocol = protocol;
        this.primaryElection = primaryElection;
        this.threadContext = threadContext;
        clusterMembershipService.addListener((EventListener)this.membershipEventListener);
        primaryElection.addListener((EventListener)this.primaryElectionListener);
        this.log = ContextualLoggerFactory.getLogger(this.getClass(), (LoggerContext)LoggerContext.builder(SessionClient.class).addValue((Object)clientName).add("type", (Object)primitiveType.name()).add("name", (Object)descriptor.name()).build());
    }

    public String name() {
        return this.descriptor.name();
    }

    public PrimitiveType type() {
        return this.primitiveType;
    }

    public ThreadContext context() {
        return this.threadContext;
    }

    public PrimitiveState getState() {
        return this.state;
    }

    public PartitionId partitionId() {
        return this.partitionId;
    }

    public SessionId sessionId() {
        return this.sessionId;
    }

    public void addStateChangeListener(Consumer<PrimitiveState> listener) {
        this.stateChangeListeners.add(listener);
    }

    public void removeStateChangeListener(Consumer<PrimitiveState> listener) {
        this.stateChangeListeners.remove(listener);
    }

    public CompletableFuture<byte[]> execute(PrimitiveOperation operation) {
        ComposableFuture future = new ComposableFuture();
        this.threadContext.execute(() -> {
            if (this.term.primary() == null) {
                this.primaryElection.getTerm().whenCompleteAsync((term, error) -> {
                    if (error == null) {
                        if (term.term() <= this.term.term() || term.primary() == null) {
                            future.completeExceptionally((Throwable)new PrimitiveException.Unavailable());
                        } else {
                            this.term = term;
                            this.execute(operation, 1, (ComposableFuture<byte[]>)future);
                        }
                    } else {
                        future.completeExceptionally((Throwable)new PrimitiveException.Unavailable());
                    }
                }, (Executor)this.threadContext);
            } else {
                this.execute(operation, 1, (ComposableFuture<byte[]>)future);
            }
        });
        return future;
    }

    private void execute(PrimitiveOperation operation, int attempt, ComposableFuture<byte[]> future) {
        if (attempt > 50) {
            future.completeExceptionally((Throwable)new PrimitiveException.Unavailable());
            return;
        }
        ExecuteRequest request = ExecuteRequest.request(this.descriptor, (Long)this.sessionId.id(), this.clusterMembershipService.getLocalMember().id(), operation);
        this.log.trace("Sending {} to {}", (Object)request, (Object)this.term.primary());
        PrimaryTerm term = this.term;
        if (term.primary() != null) {
            this.protocol.execute(term.primary().memberId(), request).whenCompleteAsync((response, error) -> {
                if (error == null) {
                    this.log.trace("Received {}", response);
                    if (response.status() == PrimaryBackupResponse.Status.OK) {
                        future.complete((Object)response.result());
                    } else if (this.term.term() > term.term()) {
                        this.execute(operation).whenComplete((BiConsumer)future);
                    } else {
                        this.primaryElection.getTerm().whenComplete((newTerm, termError) -> {
                            if (termError == null) {
                                if (newTerm.term() > term.term() && newTerm.primary() != null) {
                                    this.execute(operation).whenComplete((BiConsumer)future);
                                } else {
                                    this.threadContext.schedule(Duration.ofMillis(100L), () -> this.execute(operation, attempt + 1, future));
                                }
                            } else {
                                Throwable cause = Throwables.getRootCause((Throwable)termError);
                                if (cause instanceof PrimitiveException.Unavailable || cause instanceof TimeoutException) {
                                    this.threadContext.schedule(Duration.ofMillis(100L), () -> this.execute(operation, attempt + 1, future));
                                } else {
                                    future.completeExceptionally((Throwable)new PrimitiveException.Unavailable());
                                }
                            }
                        });
                    }
                } else if (this.term.term() > term.term()) {
                    this.execute(operation).whenComplete((BiConsumer)future);
                } else {
                    Throwable cause = Throwables.getRootCause((Throwable)error);
                    if (cause instanceof PrimitiveException.Unavailable || cause instanceof TimeoutException) {
                        this.threadContext.schedule(Duration.ofMillis(100L), () -> this.execute(operation, attempt + 1, future));
                    } else {
                        future.completeExceptionally(error);
                    }
                }
            }, (Executor)this.threadContext);
        } else {
            future.completeExceptionally((Throwable)new ConnectException());
        }
    }

    public void addEventListener(EventType eventType, Consumer<PrimitiveEvent> listener) {
        this.eventListeners.computeIfAbsent(eventType.canonicalize(), t -> Sets.newLinkedHashSet()).add(listener);
    }

    public void removeEventListener(EventType eventType, Consumer<PrimitiveEvent> listener) {
        this.eventListeners.computeIfAbsent(eventType.canonicalize(), t -> Sets.newLinkedHashSet()).remove(listener);
    }

    private void changeReplicas(PrimaryTerm term) {
        this.threadContext.execute(() -> {
            if (this.term == null || term.term() > this.term.term()) {
                this.term = term;
            }
        });
    }

    private void handleClusterEvent(ClusterMembershipEvent event) {
        PrimaryTerm term = this.term;
        if (term != null && event.type() == ClusterMembershipEvent.Type.MEMBER_REMOVED && ((Member)event.subject()).id().equals((Object)term.primary().memberId())) {
            this.threadContext.execute(() -> {
                this.state = PrimitiveState.SUSPENDED;
                this.stateChangeListeners.forEach(l -> l.accept(this.state));
            });
        }
    }

    private void handleEvent(PrimitiveEvent event) {
        this.log.trace("Received {}", (Object)event);
        Set<Consumer<PrimitiveEvent>> listeners = this.eventListeners.get(event.type());
        if (listeners != null) {
            listeners.forEach(l -> l.accept(event));
        }
    }

    public CompletableFuture<SessionClient> connect() {
        CompletableFuture<SessionClient> future = new CompletableFuture<SessionClient>();
        this.threadContext.execute(() -> this.connect(1, future));
        return future;
    }

    private void connect(int attempt, CompletableFuture<SessionClient> future) {
        if (attempt > 50) {
            future.completeExceptionally((Throwable)new PrimitiveException.Unavailable());
            return;
        }
        this.primaryElection.getTerm().whenCompleteAsync((term, error) -> {
            if (error == null) {
                if (term.primary() == null) {
                    future.completeExceptionally((Throwable)new PrimitiveException.Unavailable());
                } else {
                    this.term = term;
                    this.protocol.registerEventListener(this.sessionId, this::handleEvent, (Executor)this.threadContext);
                    future.complete(this);
                }
            } else {
                Throwable cause = Throwables.getRootCause((Throwable)error);
                if (cause instanceof PrimitiveException.Unavailable || cause instanceof TimeoutException) {
                    this.threadContext.schedule(Duration.ofMillis(100L), () -> this.connect(attempt + 1, future));
                } else {
                    future.completeExceptionally((Throwable)new PrimitiveException.Unavailable());
                }
            }
        }, (Executor)this.threadContext);
    }

    public CompletableFuture<Void> close() {
        CompletableFuture<Void> future = new CompletableFuture<Void>();
        PrimaryTerm term = this.term;
        if (term.primary() != null) {
            this.protocol.close(term.primary().memberId(), new CloseRequest(this.descriptor, (Long)this.sessionId.id())).whenCompleteAsync((response, error) -> {
                try {
                    this.protocol.unregisterEventListener(this.sessionId);
                    this.clusterMembershipService.removeListener((EventListener)this.membershipEventListener);
                    this.primaryElection.removeListener((EventListener)this.primaryElectionListener);
                }
                finally {
                    future.complete(null);
                }
            }, (Executor)this.threadContext);
        } else {
            future.complete(null);
        }
        return future;
    }

    public static abstract class Builder
    extends SessionClient.Builder {
        protected Consistency consistency = Consistency.SEQUENTIAL;
        protected Replication replication = Replication.ASYNCHRONOUS;
        protected Recovery recovery = Recovery.RECOVER;
        protected int numBackups = 1;
        protected int maxRetries = 0;
        protected Duration retryDelay = Duration.ofMillis(100L);

        public Builder withConsistency(Consistency consistency) {
            this.consistency = (Consistency)Preconditions.checkNotNull((Object)consistency, (Object)"consistency cannot be null");
            return this;
        }

        public Builder withReplication(Replication replication) {
            this.replication = (Replication)Preconditions.checkNotNull((Object)replication, (Object)"replication cannot be null");
            return this;
        }

        public Builder withRecovery(Recovery recovery) {
            this.recovery = (Recovery)Preconditions.checkNotNull((Object)recovery, (Object)"recovery cannot be null");
            return this;
        }

        public Builder withNumBackups(int numBackups) {
            Preconditions.checkArgument((numBackups >= 0 ? 1 : 0) != 0, (Object)"numBackups must be positive");
            this.numBackups = numBackups;
            return this;
        }

        public Builder withMaxRetries(int maxRetries) {
            Preconditions.checkArgument((maxRetries >= 0 ? 1 : 0) != 0, (Object)"maxRetries must be positive");
            this.maxRetries = maxRetries;
            return this;
        }

        public Builder withRetryDelayMillis(long retryDelayMillis) {
            return this.withRetryDelay(Duration.ofMillis(retryDelayMillis));
        }

        public Builder withRetryDelay(long retryDelay, TimeUnit timeUnit) {
            return this.withRetryDelay(Duration.ofMillis(timeUnit.toMillis(retryDelay)));
        }

        public Builder withRetryDelay(Duration retryDelay) {
            this.retryDelay = (Duration)Preconditions.checkNotNull((Object)retryDelay, (Object)"retryDelay cannot be null");
            return this;
        }
    }
}

