/*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/
import { Injectable } from "@angular/core";
import type {
	Capacity,
	Health,
	RequestDeliveryService,
	ResponseDeliveryService,
	ResponseDeliveryServiceSSLKey,
	SteeringConfiguration,
	TypeFromResponse
} from "trafficops-types";

import type {
	DataPoint,
	DataSetWithSummary,
	TPSData,
} from "src/app/models";

import { CDNService, ProfileService, TypeService, UserService } from "..";

/**
 * The type of a raw response returned from the API that has to be massaged
 * into a DataSet.
 */
interface DataResponse {
	series: {
		name: string;
		values: Array<[number, number | null]>;
	};
	summary?: {
		min: number;
		max: number;
		fifthPercentile: number;
		ninetyFifthPercentile: number;
		ninetyEightPercentile: number;
		mean: number;
	};
}

/**
 * A random dataset (values [0-100]) generated by {@link generateDataSet}.
 */
interface GeneratedDataSet {
	data: [number, number][];
	max: number;
	mean: number;
	min: number;
}

/**
 * Generates random bandwidth or tps data for presenting as API output in tests.
 *
 * @param start The start time of the data set.
 * @param end The end time of the data set.
 * @param step The amount of time within which to group data into points.
 * @returns A generated data set, including some summary metrics.
 */
function generateDataSet(start: Date, end: Date, step: number): GeneratedDataSet {
	const data = new Array<[number, number]>();
	const finish = end.valueOf();
	let min = Infinity;
	let max = -Infinity;
	let sum = 0;
	for (let current = start.valueOf(); current < finish; current += step) {
		const value = Math.random()*100;
		if (value < min) {
			min = value;
		}
		if (value > max) {
			max = value;
		}
		sum += value;
		data.push([current, value]);
	}
	return {
		data,
		max,
		mean: sum / data.length,
		min
	};
}

/**
 * DeliveryServiceService exposes API functionality related to Delivery Services.
 */
@Injectable()
export class DeliveryServiceService {

	private readonly deliveryServices = new Array<ResponseDeliveryService>();
	private idCounter = 0;
	private readonly dsTypes = [
		{
			description: "No Content Routing - arbitrary remap at the edge, no Traffic Router config",
			id: 5,
			lastUpdated: new Date(),
			name: "ANY_MAP",
			useInTable: "deliveryservice"
		},
		{
			description: "Client-Controlled Steering Delivery Service",
			id: 6,
			lastUpdated: new Date(),
			name: "CLIENT_STEERING",
			useInTable: "deliveryservice"
		},
		{
			description: "DNS Content Routing",
			id: 7,
			lastUpdated: new Date(),
			name: "DNS",
			useInTable: "deliveryservice"
		},
		{
			description: "DNS Content routing, RAM cache, Local",
			id: 8,
			lastUpdated: new Date(),
			name: "DNS_LIVE",
			useInTable: "deliveryservice"
		},
		{
			description: "DNS Content routing, RAM cache, National",
			id: 9,
			lastUpdated: new Date(),
			name: "DNS_LIVE_NATNL",
			useInTable: "deliveryservice"
		},
		{
			description: "HTTP Content Routing",
			id: 10,
			lastUpdated: new Date(),
			name: "HTTP",
			useInTable: "deliveryservice"
		},
		{
			description: "HTTP Content routing cache in RAM",
			id: 11,
			lastUpdated: new Date(),
			name: "HTTP_LIVE",
			useInTable: "deliveryservice"
		},
		{
			description: "HTTP Content routing, RAM cache, National",
			id: 12,
			lastUpdated: new Date(),
			name: "HTTP_LIVE_NATNL",
			useInTable: "deliveryservice"
		},
		{
			description: "HTTP Content Routing, no caching",
			id: 13,
			lastUpdated: new Date(),
			name: "HTTP_NO_CACHE",
			useInTable: "deliveryservice"
		},
		{
			description: "Steering Delivery Service",
			id: 14,
			lastUpdated: new Date(),
			name: "STEERING",
			useInTable: "deliveryservice"
		}
	];
	private readonly dsSSLKeys: Array<ResponseDeliveryServiceSSLKey> = [{
		cdn: "'",
		certificate: {crt: "", csr: "", key: ""},
		deliveryservice: "xml",
		expiration: new Date(),
		version: ""
	}];

	constructor(
		private readonly cdnService: CDNService,
		private readonly profileService: ProfileService,
		private readonly userService: UserService,
		private readonly typeService: TypeService
	) {}

	/**
	 * Gets a list of all Steering Configurations
	 *
	 * @returns An array of Steering Configurations
	 */
	public async getSteering(): Promise<Array<SteeringConfiguration>> {
		return [];
	}

	public async getDeliveryServices(id: string | number): Promise<ResponseDeliveryService>;
	public async getDeliveryServices(): Promise<Array<ResponseDeliveryService>>;
	/**
	 * Gets a list of all visible Delivery Services
	 *
	 * @param id A unique identifier for a Delivery Service - either a numeric id or an "xml_id"
	 * @throws TypeError if ``id`` is not a proper type
	 * @returns An array of `DeliveryService` objects.
	 */
	public async getDeliveryServices(id?: string | number): Promise<ResponseDeliveryService[] | ResponseDeliveryService> {
		if (id !== undefined) {
			let ds;
			switch (typeof id) {
				case "string":
					ds = this.deliveryServices.filter(d=>d.xmlId === id)[0];
					break;
				case "number":
					ds = this.deliveryServices.filter(d=>d.id === id);
			}
			if (!ds) {
				throw new Error(`no such Delivery Service: ${id}`);
			}
			return ds;
		}
		return this.deliveryServices;
	}

	/**
	 * Creates a new Delivery Service
	 *
	 * @param ds The new Delivery Service object
	 * @returns A boolean value indicating the success of the operation
	 */
	public async createDeliveryService(ds: RequestDeliveryService): Promise<ResponseDeliveryService> {
		const cdn = await this.cdnService.getCDNs(ds.cdnId);
		let profile = null;
		if (ds.profileId !== null && ds.profileId !== undefined) {
			profile = await this.profileService.getProfiles(ds.profileId);
		}
		const tenant = await this.userService.getTenants(ds.tenantId);
		const type = await this.typeService.getTypes(ds.typeId);
		let created: ResponseDeliveryService = {
			...ds,
			anonymousBlockingEnabled: ds.anonymousBlockingEnabled ?? false,
			ccrDnsTtl: ds.ccrDnsTtl ?? null,
			cdnName: cdn.name,
			checkPath: ds.checkPath ?? null,
			consistentHashQueryParams: null,
			consistentHashRegex: ds.consistentHashRegex ?? null,
			deepCachingType: ds.deepCachingType ?? "NEVER",
			dnsBypassCname: ds.dnsBypassCname ?? null,
			dnsBypassIp: ds.dnsBypassIp ?? null,
			dnsBypassIp6: ds.dnsBypassIp6 ?? null,
			dnsBypassTtl: ds.dnsBypassTtl ? Number(ds.dnsBypassTtl) : null,
			ecsEnabled: ds.ecsEnabled ?? false,
			edgeHeaderRewrite: ds.edgeHeaderRewrite ?? null,
			exampleURLs: [
				`https://${ds.routingName ?? "cdn"}.${ds.xmlId}.${cdn.name}.${cdn.domainName}`
			],
			firstHeaderRewrite: ds.firstHeaderRewrite ?? null,
			fqPacingRate: ds.fqPacingRate ?? 0,
			geoLimitCountries: ds.geoLimitCountries ?? null,
			geoLimitRedirectURL: ds.geoLimitRedirectUrl ?? null,
			globalMaxMbps: ds.globalMaxMbps ?? null,
			globalMaxTps: ds.globalMaxTps ?? null,
			id: ++this.idCounter,
			initialDispersion: ds.initialDispersion || 1,
			innerHeaderRewrite: ds.innerHeaderRewrite ?? null,
			ipv6RoutingEnabled: ds.ipv6RoutingEnabled ?? false,
			lastHeaderRewrite: ds.lastHeaderRewrite ?? null,
			lastUpdated: new Date(),
			longDesc: ds.longDesc ?? null,
			longDesc1: ds.longDesc1 ?? undefined,
			longDesc2: ds.longDesc2 ?? undefined,
			matchList: [
				{
					pattern: `.*\\\\.${ds.xmlId}\\\\..*`,
					setNumber: 0,
					type: "HOST_REGEX",
				}
			],
			maxDnsAnswers: ds.maxDnsAnswers ?? null,
			maxOriginConnections: ds.maxOriginConnections ?? 0,
			maxRequestHeaderBytes: ds.maxRequestHeaderBytes ?? 0,
			midHeaderRewrite: ds.midHeaderRewrite ?? null,
			missLat: ds.missLat ?? 0,
			missLong: ds.missLat ?? 0,
			multiSiteOrigin: ds.multiSiteOrigin ?? false,
			orgServerFqdn: ds.orgServerFqdn ?? null,
			originShield: ds.originShield ?? null,
			profileDescription: null,
			profileId: null,
			profileName: null,
			protocol: ds.protocol ?? null,
			qstringIgnore: ds.qstringIgnore ?? null,
			rangeRequestHandling: ds.rangeRequestHandling ?? null,
			rangeSliceBlockSize: ds.rangeSliceBlockSize ?? null,
			regexRemap: ds.regexRemap ?? null,
			routingName: ds.routingName || "cdn",
			serviceCategory: ds.serviceCategory ?? null,
			signed: ds.signed ?? false,
			signingAlgorithm: ds.signingAlgorithm ?? "uri_signing",
			sslKeyVersion: ds.sslKeyVersion ?? null,
			tenant: tenant.name,
			tlsVersions: ds.tlsVersions && ds.tlsVersions.length > 0 ? ds.tlsVersions : null,
			topology: ds.topology ?? null,
			trRequestHeaders: ds.trRequestHeaders ?? null,
			trResponseHeaders: ds.trResponseHeaders ?? null,
			type: type.name
		};
		if (ds.consistentHashQueryParams && ds.consistentHashQueryParams.length > 0) {
			created.consistentHashQueryParams = [ds.consistentHashQueryParams[0], ...ds.consistentHashQueryParams.slice(1)];
		}
		if (profile) {
			created = {
				...created,
				profileDescription: profile.description,
				profileId: profile.id,
				profileName: profile.name,
			};
		}
		this.deliveryServices.push(created);
		return created;
	}

	/**
	 * Retrieves capacity statistics for the Delivery Service identified by a given, unique,
	 * integral value.
	 *
	 * @param d Either a {@link ResponseDeliveryService} or an integral, unique identifier of a Delivery Service
	 * @returns An object that hopefully has the right keys to represent capacity.
	 * @throws If `d` is a {@link ResponseDeliveryService} that has no (valid) id
	 */
	public async getDSCapacity(d: number | ResponseDeliveryService): Promise<Capacity> {
		let id: number;
		if (typeof d === "number") {
			id = d;
		} else {
			if (!d.id || d.id < 0) {
				throw new Error("Delivery Service id must be defined!");
			}
			id = d.id;
		}

		const ds = this.deliveryServices.filter(service=>service.id === id)[0];
		if (!ds) {
			throw new Error(`no such Delivery Service: #${id}`);
		}
		const val = ds.lastUpdated ? ds.lastUpdated.valueOf() : 100;

		return {
			availablePercent: val %40,
			maintenancePercent: val %40,
			unavailablePercent: val %10,
			utilizedPercent: val %20
		};
	}

	/**
	 * Retrieves the Cache Group health of a Delivery Service identified by a given, unique,
	 * integral value.
	 *
	 * @param d The integral, unique identifier of a Delivery Service
	 * @returns A response from the health endpoint
	 */
	public async getDSHealth(d: number): Promise<Health> {
		const ds = this.deliveryServices.filter(service => service.id === d)[0];
		if (!ds) {
			throw new Error(`no such Delivery Service: #${d}`);
		}
		const val = ds.lastUpdated ? ds.lastUpdated.valueOf() : 100;
		return {
			cacheGroups: [],
			totalOffline: val % 50,
			totalOnline: 100-(val%50)
		};
	}

	public async getDSKBPS(d: string, s: Date, e: Date, i: string, u: boolean, dataOnly: true): Promise<Array<DataPoint>>;
	public async getDSKBPS(d: string, start: Date, end: Date, interval: string, useMids: boolean, dataOnly?: false): Promise<DataResponse>;
	/**
	 * Retrieves Delivery Service throughput statistics for a given time period, averaged over a given
	 * interval.
	 *
	 * @param d The `xml_id` of a Delivery Service
	 * @param start A date/time from which to start data collection
	 * @param end A date/time at which to end data collection
	 * @param interval A unit-suffixed interval over which data will be "binned"
	 * @param _ Unuzed - kept for compatibility with the "concrete" service.
	 * @param dataOnly Only returns the data series, not any supplementing meta info found in the API response
	 * @returns An Array of datapoint Arrays (length 2 containing a date string and data value)
	 */
	public async getDSKBPS(
		d: string,
		start: Date,
		end: Date,
		interval: string,
		_: boolean,
		dataOnly?: boolean
	): Promise<Array<DataPoint> | DataResponse> {
		const ds = this.deliveryServices.filter(service=>service.xmlId === d)[0];
		if (!ds) {
			throw new Error(`no such Delivery Service: ${d}`);
		}
		// Here we assume that `interval` is a number followed by 'm' for 'minutes'.
		const step = parseInt(interval.slice(0, -1), 10)*60000;

		const {data, max, mean, min} = generateDataSet(start, end, step);

		if (dataOnly) {
			return data.map(p=>({t: new Date(p[0]), y: p[1]}));
		}
		return {
			series: {
				name: `kbps.ds.${step / 60000}min`,
				values: data
			},
			summary: {
				// TODO: these percentiles should technically be accurate
				// (depending on the implementation of Math.random) but could
				// probably be calculated. If that would ever matter.
				fifthPercentile: 5,
				max,
				mean,
				min,
				ninetyEightPercentile: 98,
				ninetyFifthPercentile: 95,
			}
		};
	}

	/**
	 * Gets total TPS data for a Delivery Service. To get TPS data broken down by HTTP status, use {@link getAllDSTPSData}.
	 *
	 * @param d The name (xmlid) of the Delivery Service for which TPS stats will be fetched
	 * @param start The desired start date/time of the data range (must not have nonzero milliseconds!)
	 * @param end The desired end date/time of the data range (must not have nonzero milliseconds!)
	 * @param interval A string that describes the interval across which to 'bucket' data e.g. '60s'
	 * @returns The requested DataResponse.
	 */
	public async getDSTPS(
		d: string,
		start: Date,
		end: Date,
		interval: string,
	): Promise<DataResponse> {
		const ds = this.deliveryServices.filter(service=>service.xmlId === d)[0];
		if (!ds) {
			throw new Error(`no such Delivery Service: ${d}`);
		}
		// Here we assume that `interval` is a number followed by 'm' for 'minutes'.
		const step = parseInt(interval.slice(0, -1), 10)*60000;

		const {data, max, mean, min} = generateDataSet(start, end, step);

		return {
			series: {
				name: `tps_total.ds.${step / 60000}min`,
				values: data
			},
			summary: {
				// TODO: these percentiles should technically be accurate
				// (depending on the implementation of Math.random) but could
				// probably be calculated. If that would ever matter.
				fifthPercentile: 5,
				max,
				mean,
				min,
				ninetyEightPercentile: 98,
				ninetyFifthPercentile: 95,
			}
		};
	}

	/**
	 * Gets total TPS data for a Delivery Service, as well as TPS data by HTTP response type.
	 *
	 * @param d The name (xmlid) of the Delivery Service for which TPS stats will be fetched
	 * @param start The desired start date/time of the data range (must not have nonzero milliseconds!)
	 * @param end The desired end date/time of the data range (must not have nonzero milliseconds!)
	 * @param interval A string that describes the interval across which to 'bucket' data e.g. '60s'
	 * @returns The requested TPSData.
	 */
	public async getAllDSTPSData(
		d: string,
		start: Date,
		end: Date,
		interval: string,
	): Promise<TPSData> {
		const ds = this.deliveryServices.filter(service=>service.xmlId === d)[0];
		if (!ds) {
			throw new Error(`no such Delivery Service: ${d}`);
		}

		// Here we assume that `interval` is a number followed by 'm' for 'minutes'.
		const step = parseInt(interval.slice(0, -1), 10)*60000;

		const {data, max, mean, min} = generateDataSet(start, end, step);
		const total = {
			dataSet: {
				data: data.map(p=>({t: new Date(p[0]), y: p[1]})),
				label: "tps_total"
			},
			fifthPercentile: 5,
			max,
			mean,
			min,
			ninetyEighthPercentile: 98,
			ninetyFifthPercentile: 95
		};
		const mkDataSet = (label: string): DataSetWithSummary => ({
			dataSet: {
				data: total.dataSet.data.map(p=>({t: p.t, y: p.y/4})),
				label
			},
			fifthPercentile: total.fifthPercentile / 4,
			max: Math.random()*4 >= 3 ? total.max : total.max / 4,
			mean: total.mean / 4,
			min: Math.random()*4 >=3 ? total.min : (total.min + 7 > total.max ? total.min : total.min + 7) ,
			ninetyEighthPercentile: total.ninetyEighthPercentile / 4,
			ninetyFifthPercentile: total.ninetyFifthPercentile / 4
		});

		const success = mkDataSet("tps_2xx");
		const redirection = mkDataSet("tps_3xx");
		const clientError = mkDataSet("tps_4xx");
		const serverError = mkDataSet("tps_5xx");

		return {
			clientError,
			redirection,
			serverError,
			success,
			total,
		};
	}

	/**
	 * This method is handled seperately from :js:method:`APIService.getTypes` because this information
	 * (should) never change, and therefore can be cached. This method makes an HTTP request iff the values are not already
	 * cached.
	 *
	 * @returns An array of all of the Type objects in Traffic Ops that refer specifically to Delivery Service
	 * 	types.
	 */
	public async getDSTypes(): Promise<Array<TypeFromResponse>> {
		return this.dsTypes;
	}

	/**
	 * Gets a Delivery Service's SSL Keys
	 *
	 * @param ds The delivery service xmlid or object
	 * @returns The DS ssl keys
	 */
	public async getSSLKeys(ds: string | ResponseDeliveryService): Promise<ResponseDeliveryServiceSSLKey> {
		const xmlId = typeof ds === "string" ? ds : ds.xmlId;
		const key = this.dsSSLKeys.find(k => k.deliveryservice === xmlId);
		if(!key) {
			throw new Error(`no such Delivery Service: ${xmlId}`);
		}

		return key;
	}
}
