/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.debug.replicas;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import org.apache.commons.lang3.time.DurationFormatUtils;
import org.apache.hadoop.hdds.client.ReplicationConfig;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.scm.cli.ScmOption;
import org.apache.hadoop.hdds.server.JsonUtils;
import org.apache.hadoop.ozone.client.ObjectStore;
import org.apache.hadoop.ozone.client.OzoneBucket;
import org.apache.hadoop.ozone.client.OzoneClient;
import org.apache.hadoop.ozone.client.OzoneClientException;
import org.apache.hadoop.ozone.client.OzoneKey;
import org.apache.hadoop.ozone.client.OzoneVolume;
import org.apache.hadoop.ozone.debug.replicas.BlockExistenceVerifier;
import org.apache.hadoop.ozone.debug.replicas.BlockVerificationResult;
import org.apache.hadoop.ozone.debug.replicas.ChecksumVerifier;
import org.apache.hadoop.ozone.debug.replicas.ContainerStateVerifier;
import org.apache.hadoop.ozone.debug.replicas.ReplicaVerifier;
import org.apache.hadoop.ozone.om.helpers.OmKeyInfo;
import org.apache.hadoop.ozone.om.helpers.OmKeyLocationInfo;
import org.apache.hadoop.ozone.shell.Handler;
import org.apache.hadoop.ozone.shell.OzoneAddress;
import org.apache.hadoop.ozone.shell.ShellReplicationOptions;
import org.apache.hadoop.ozone.util.ShutdownHookManager;
import picocli.CommandLine;

@CommandLine.Command(name="verify", description={"Run checks to verify data across replicas. By default prints only the keys with failed checks. Optionally you can filter keys by replication type (RATIS, EC) and factor."})
public class ReplicasVerify
extends Handler {
    @CommandLine.Mixin
    private ScmOption scmOption;
    @CommandLine.Mixin
    private ShellReplicationOptions replication;
    @CommandLine.Parameters(arity="1", description={"Ozone URI could either be a full URI or short URI.\nFull URI should start with o3://, in case of non-HA\nclusters it should be followed by the host name and\noptionally the port number. In case of HA clusters\nthe service id should be used. Service id provides a\nlogical name for multiple hosts and it is defined\nin the property ozone.om.service.ids.\nExample of a full URI with host name and port number\nfor a key:\no3://omhostname:9862/vol1/bucket1/key1\nWith a service id for a volume:\no3://omserviceid/vol1/\nShort URI should start from the volume.\nExample of a short URI for a bucket:\nvol1/bucket1\nAny unspecified information will be identified from\nthe config files.\n"})
    private String uri;
    @CommandLine.Option(names={"--all-results"}, description={"Print results for all passing and failing keys"})
    private boolean allResults;
    @CommandLine.ArgGroup(exclusive=false, multiplicity="1")
    private Verification verification;
    @CommandLine.Option(names={"--container-cache-size"}, description={"Size (in number of containers) of the in-memory cache for container state verification '--container-state'. Default is 1 million containers (which takes around 43MB). Value must be greater than zero, otherwise the default of 1 million is considered. Note: This option is ignored if '--container-state' option is not used."}, defaultValue="1000000")
    private long containerCacheSize;
    private List<ReplicaVerifier> replicaVerifiers;
    private static final String DURATION_FORMAT = "HH:mm:ss,SSS";
    private long startTime;
    private long endTime;
    private String verificationScope;
    private final List<String> verificationTypes = new ArrayList<String>();
    private final AtomicInteger volumesProcessed = new AtomicInteger(0);
    private final AtomicInteger bucketsProcessed = new AtomicInteger(0);
    private final AtomicInteger keysProcessed = new AtomicInteger(0);
    private final AtomicInteger keysPassed = new AtomicInteger(0);
    private final AtomicInteger keysFailed = new AtomicInteger(0);
    private final Map<String, AtomicInteger> failuresByType = new ConcurrentHashMap<String, AtomicInteger>();
    private volatile Throwable exception;

    private void addVerifier(boolean condition, Supplier<ReplicaVerifier> verifierSupplier) {
        if (condition) {
            ReplicaVerifier verifier = verifierSupplier.get();
            this.replicaVerifiers.add(verifier);
            String verifierType = verifier.getType();
            this.verificationTypes.add(verifierType);
            this.failuresByType.put(verifierType, new AtomicInteger(0));
        }
    }

    protected void execute(OzoneClient client, OzoneAddress address) throws IOException {
        this.startTime = System.nanoTime();
        this.verificationScope = !address.getKeyName().isEmpty() ? "Key" : (!address.getBucketName().isEmpty() ? "Bucket" : (!address.getVolumeName().isEmpty() ? "Volume" : "All Volumes"));
        this.replicaVerifiers = new ArrayList<ReplicaVerifier>();
        this.addVerifier(this.verification.doExecuteChecksums, () -> {
            try {
                return new ChecksumVerifier(this.getConf());
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        });
        this.addVerifier(this.verification.doExecuteBlockExistence, () -> {
            try {
                return new BlockExistenceVerifier(this.getConf());
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        });
        this.addVerifier(this.verification.doExecuteReplicaState, () -> {
            try {
                return new ContainerStateVerifier(this.getConf(), this.containerCacheSize);
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        });
        this.addShutdownHook();
        try {
            this.findCandidateKeys(client, address);
        }
        catch (Exception e) {
            this.exception = e;
            throw e;
        }
        finally {
            this.endTime = System.nanoTime();
        }
    }

    protected OzoneAddress getAddress() throws OzoneClientException {
        return new OzoneAddress(this.uri);
    }

    void findCandidateKeys(OzoneClient ozoneClient, OzoneAddress address) throws IOException {
        ObjectStore objectStore = ozoneClient.getObjectStore();
        String volumeName = address.getVolumeName();
        String bucketName = address.getBucketName();
        String keyName = address.getKeyName();
        ObjectNode root = JsonUtils.createObjectNode(null);
        ArrayNode keysArray = root.putArray("keys");
        AtomicBoolean allKeysPassed = new AtomicBoolean(true);
        if (!keyName.isEmpty()) {
            this.processKey(ozoneClient, volumeName, bucketName, keyName, keysArray, allKeysPassed);
        } else if (!bucketName.isEmpty()) {
            OzoneVolume volume = objectStore.getVolume(volumeName);
            OzoneBucket bucket = volume.getBucket(bucketName);
            this.checkBucket(ozoneClient, bucket, keysArray, allKeysPassed);
        } else if (!volumeName.isEmpty()) {
            OzoneVolume volume = objectStore.getVolume(volumeName);
            this.checkVolume(ozoneClient, volume, keysArray, allKeysPassed);
        } else {
            Iterator it = objectStore.listVolumes(null);
            while (it.hasNext()) {
                this.checkVolume(ozoneClient, (OzoneVolume)it.next(), keysArray, allKeysPassed);
            }
        }
        root.put("pass", allKeysPassed.get());
        System.out.println(JsonUtils.toJsonStringWithDefaultPrettyPrinter((Object)root));
    }

    void checkVolume(OzoneClient ozoneClient, OzoneVolume volume, ArrayNode keysArray, AtomicBoolean allKeysPassed) throws IOException {
        this.volumesProcessed.incrementAndGet();
        Iterator it = volume.listBuckets(null);
        while (it.hasNext()) {
            OzoneBucket bucket = (OzoneBucket)it.next();
            this.checkBucket(ozoneClient, bucket, keysArray, allKeysPassed);
        }
    }

    void checkBucket(OzoneClient ozoneClient, OzoneBucket bucket, ArrayNode keysArray, AtomicBoolean allKeysPassed) throws IOException {
        this.bucketsProcessed.incrementAndGet();
        Iterator it = bucket.listKeys(null);
        while (it.hasNext()) {
            OzoneKey key = (OzoneKey)it.next();
            if (key.getName().endsWith("/")) continue;
            this.processKey(ozoneClient, key.getVolumeName(), key.getBucketName(), key.getName(), keysArray, allKeysPassed);
        }
    }

    void processKey(OzoneClient ozoneClient, String volumeName, String bucketName, String keyName, ArrayNode keysArray, AtomicBoolean allKeysPassed) throws IOException {
        this.keysProcessed.incrementAndGet();
        OmKeyInfo keyInfo = ozoneClient.getProxy().getKeyInfo(volumeName, bucketName, keyName, false);
        if (!this.shouldProcessKeyByReplicationType(keyInfo)) {
            return;
        }
        ObjectNode keyNode = JsonUtils.createObjectNode(null);
        keyNode.put("volumeName", volumeName);
        keyNode.put("bucketName", bucketName);
        keyNode.put("name", keyName);
        ArrayNode blocksArray = keyNode.putArray("blocks");
        boolean keyPass = true;
        HashSet<String> failedVerificationTypes = new HashSet<String>();
        for (OmKeyLocationInfo keyLocation : keyInfo.getLatestVersionLocations().getBlocksLatestVersionOnly()) {
            long containerID = keyLocation.getContainerID();
            long localID = keyLocation.getLocalID();
            ObjectNode blockNode = blocksArray.addObject();
            blockNode.put("containerID", containerID);
            blockNode.put("blockID", localID);
            ArrayNode replicasArray = blockNode.putArray("replicas");
            boolean blockPass = true;
            for (DatanodeDetails datanode : keyLocation.getPipeline().getNodes()) {
                ObjectNode replicaNode = replicasArray.addObject();
                ObjectNode datanodeNode = replicaNode.putObject("datanode");
                datanodeNode.put("uuid", datanode.getUuidString());
                datanodeNode.put("hostname", datanode.getHostName());
                ArrayNode checksArray = replicaNode.putArray("checks");
                boolean replicaPass = true;
                int replicaIndex = keyLocation.getPipeline().getReplicaIndex(datanode);
                for (ReplicaVerifier verifier : this.replicaVerifiers) {
                    BlockVerificationResult result = verifier.verifyBlock(datanode, keyLocation);
                    ObjectNode checkNode = checksArray.addObject();
                    checkNode.put("type", verifier.getType());
                    checkNode.put("completed", result.isCompleted());
                    checkNode.put("pass", result.passed());
                    ArrayNode failuresArray = checkNode.putArray("failures");
                    for (String failure : result.getFailures()) {
                        failuresArray.addObject().put("message", failure);
                    }
                    replicaNode.put("replicaIndex", replicaIndex);
                    if (result.passed()) continue;
                    replicaPass = false;
                    failedVerificationTypes.add(verifier.getType());
                }
                if (replicaPass) continue;
                blockPass = false;
            }
            if (blockPass) continue;
            keyPass = false;
        }
        keyNode.put("pass", keyPass);
        if (keyPass) {
            this.keysPassed.incrementAndGet();
        } else {
            this.keysFailed.incrementAndGet();
            allKeysPassed.set(false);
            failedVerificationTypes.forEach(failedType -> this.failuresByType.computeIfAbsent((String)failedType, k -> new AtomicInteger(0)).incrementAndGet());
        }
        if (!keyPass || this.allResults) {
            keysArray.add((JsonNode)keyNode);
        }
    }

    private void addShutdownHook() {
        ShutdownHookManager.get().addShutdownHook(() -> {
            if (this.endTime == 0L) {
                this.endTime = System.nanoTime();
            }
            this.printSummary(System.err);
        }, 10);
    }

    void printSummary(PrintStream out) {
        if (this.endTime == 0L) {
            this.endTime = System.nanoTime();
        }
        long execTimeNanos = this.endTime - this.startTime;
        String execTime = DurationFormatUtils.formatDuration((long)TimeUnit.NANOSECONDS.toMillis(execTimeNanos), (String)DURATION_FORMAT);
        long totalKeysProcessed = this.keysProcessed.get();
        long totalKeysPassed = this.keysPassed.get();
        long totalKeysFailed = this.keysFailed.get();
        out.println();
        out.println("***************************************************");
        out.println("REPLICA VERIFICATION SUMMARY");
        out.println("***************************************************");
        out.println("Status: " + (this.exception != null ? "Failed" : (totalKeysFailed == 0L ? "Success" : "Completed with failures")));
        out.println("Verification Scope: " + this.verificationScope);
        out.println("Verification Types: " + String.join((CharSequence)", ", this.verificationTypes));
        out.println("URI: " + this.uri);
        out.println();
        out.println("Number of Volumes processed: " + this.volumesProcessed.get());
        out.println("Number of Buckets processed: " + this.bucketsProcessed.get());
        out.println("Number of Keys processed: " + totalKeysProcessed);
        out.println();
        out.println("Keys passed verification: " + totalKeysPassed);
        out.println("Keys failed verification: " + totalKeysFailed);
        if (!this.failuresByType.isEmpty() && totalKeysFailed > 0L) {
            out.println();
            for (String verificationType : this.verificationTypes) {
                long typeFailures = this.failuresByType.get(verificationType).get();
                if (typeFailures <= 0L) continue;
                out.println("Keys failed " + verificationType + " verification: " + typeFailures);
            }
            out.println("Note: A key may fail multiple verification types, so total may exceed overall failures.");
        }
        out.println();
        out.println("Total Execution time: " + execTime);
        if (this.exception != null) {
            out.println();
            out.println("Exception: " + this.exception.getClass().getSimpleName() + ": " + this.exception.getMessage());
        }
        out.println("***************************************************");
    }

    private boolean shouldProcessKeyByReplicationType(OmKeyInfo keyInfo) {
        Optional filterConfig = this.replication.fromParams((ConfigurationSource)this.getConf());
        if (!filterConfig.isPresent()) {
            return true;
        }
        ReplicationConfig keyReplicationConfig = keyInfo.getReplicationConfig();
        ReplicationConfig filter = (ReplicationConfig)filterConfig.get();
        return keyReplicationConfig.getReplicationType().equals((Object)filter.getReplicationType()) && keyReplicationConfig.getReplication().equals(filter.getReplication());
    }

    static class Verification {
        @CommandLine.Option(names={"--checksums"}, description={"Do client side data checksum validation of all replicas."}, defaultValue="false")
        private boolean doExecuteChecksums;
        @CommandLine.Option(names={"--block-existence"}, description={"Check for block existence on datanodes."}, defaultValue="false")
        private boolean doExecuteBlockExistence;
        @CommandLine.Option(names={"--container-state"}, description={"Check the container and replica states. Containers must be in [OPEN, CLOSING, QUASI_CLOSED, CLOSED] states, and it's replicas must be in [OPEN, CLOSING, QUASI_CLOSED, CLOSED] states to pass the check. Any other states will fail the verification."}, defaultValue="false")
        private boolean doExecuteReplicaState;

        Verification() {
        }
    }
}

