/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.container.replication;

import com.google.common.collect.ImmutableList;
import java.io.IOException;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.OptionalLong;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.conf.StorageUnit;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.scm.PlacementPolicy;
import org.apache.hadoop.hdds.scm.container.ContainerInfo;
import org.apache.hadoop.hdds.scm.container.ContainerReplica;
import org.apache.hadoop.hdds.scm.container.replication.CommandTargetOverloadedException;
import org.apache.hadoop.hdds.scm.container.replication.ContainerHealthResult;
import org.apache.hadoop.hdds.scm.container.replication.ContainerReplicaOp;
import org.apache.hadoop.hdds.scm.container.replication.RatisContainerReplicaCount;
import org.apache.hadoop.hdds.scm.container.replication.ReplicationManager;
import org.apache.hadoop.hdds.scm.container.replication.ReplicationManagerMetrics;
import org.apache.hadoop.hdds.scm.container.replication.ReplicationManagerUtil;
import org.apache.hadoop.hdds.scm.container.replication.UnhealthyReplicationHandler;
import org.apache.hadoop.hdds.scm.exceptions.SCMException;
import org.apache.hadoop.hdds.scm.node.states.NodeNotFoundException;
import org.apache.hadoop.hdds.scm.pipeline.InsufficientDatanodesException;
import org.apache.hadoop.ozone.protocol.commands.ReplicateContainerCommand;
import org.apache.hadoop.ozone.protocol.commands.SCMCommand;
import org.apache.ratis.protocol.exceptions.NotLeaderException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RatisUnderReplicationHandler
implements UnhealthyReplicationHandler {
    private static final Logger LOG = LoggerFactory.getLogger(RatisUnderReplicationHandler.class);
    private final PlacementPolicy placementPolicy;
    private final long currentContainerSize;
    private final ReplicationManager replicationManager;
    private final ReplicationManagerMetrics metrics;

    public RatisUnderReplicationHandler(PlacementPolicy placementPolicy, ConfigurationSource conf, ReplicationManager replicationManager) {
        this.placementPolicy = placementPolicy;
        this.currentContainerSize = (long)conf.getStorageSize("ozone.scm.container.size", "5GB", StorageUnit.BYTES);
        this.replicationManager = replicationManager;
        this.metrics = replicationManager.getMetrics();
    }

    @Override
    public int processAndSendCommands(Set<ContainerReplica> replicas, List<ContainerReplicaOp> pendingOps, ContainerHealthResult result, int minHealthyForMaintenance) throws IOException {
        List<DatanodeDetails> targetDatanodes;
        ContainerHealthResult.UnderReplicatedHealthResult underReplicatedResult;
        ContainerInfo containerInfo = result.getContainerInfo();
        LOG.debug("Handling under replicated Ratis container {}", (Object)containerInfo);
        RatisContainerReplicaCount withUnhealthy = new RatisContainerReplicaCount(containerInfo, replicas, pendingOps, minHealthyForMaintenance, true);
        RatisContainerReplicaCount withoutUnhealthy = new RatisContainerReplicaCount(containerInfo, replicas, pendingOps, minHealthyForMaintenance, false);
        if (result instanceof ContainerHealthResult.UnderReplicatedHealthResult && (underReplicatedResult = (ContainerHealthResult.UnderReplicatedHealthResult)result).hasVulnerableUnhealthy()) {
            return this.handleVulnerableUnhealthyReplicas(withUnhealthy, pendingOps);
        }
        RatisContainerReplicaCount replicaCount = this.verifyUnderReplication(withUnhealthy, withoutUnhealthy);
        if (replicaCount == null) {
            return 0;
        }
        List<DatanodeDetails> sourceDatanodes = this.getSources(replicaCount, pendingOps);
        if (sourceDatanodes.isEmpty()) {
            LOG.warn("Cannot replicate container {} because no CLOSED, QUASI_CLOSED or UNHEALTHY replicas were found.", (Object)containerInfo);
            return 0;
        }
        try {
            targetDatanodes = this.getTargets(replicaCount, pendingOps);
        }
        catch (SCMException e) {
            SCMException.ResultCodes code = e.getResult();
            if (code != SCMException.ResultCodes.FAILED_TO_FIND_SUITABLE_NODE) {
                throw e;
            }
            LOG.warn("Cannot replicate container {} because no suitable targets were found.", (Object)containerInfo);
            this.removeUnhealthyReplicaIfPossible(containerInfo, replicas, pendingOps);
            throw e;
        }
        int commandsSent = this.sendReplicationCommands(containerInfo, sourceDatanodes, targetDatanodes);
        if (targetDatanodes.size() < replicaCount.additionalReplicaNeeded()) {
            LOG.debug("Placement policy failed to find enough targets to satisfy under replication for container {}. Targets found: {}, additional replicas needed: {}", new Object[]{containerInfo, targetDatanodes.size(), replicaCount.additionalReplicaNeeded()});
            this.metrics.incrPartialReplicationTotal();
            throw new InsufficientDatanodesException(replicaCount.additionalReplicaNeeded(), targetDatanodes.size());
        }
        return commandsSent;
    }

    private int handleVulnerableUnhealthyReplicas(RatisContainerReplicaCount replicaCount, List<ContainerReplicaOp> pendingOps) throws NotLeaderException, CommandTargetOverloadedException, SCMException {
        ContainerInfo container = replicaCount.getContainer();
        List<ContainerReplica> vulnerableUnhealthy = replicaCount.getVulnerableUnhealthyReplicas(dn -> {
            try {
                return this.replicationManager.getNodeStatus((DatanodeDetails)dn);
            }
            catch (NodeNotFoundException e) {
                LOG.warn("Exception for datanode {} while handling vulnerable replicas for container {}, with all replicas {}.", new Object[]{dn, container, replicaCount.getReplicas(), e});
                return null;
            }
        });
        LOG.info("Handling vulnerable UNHEALTHY replicas {} for container {}.", vulnerableUnhealthy, (Object)container);
        int pendingAdds = 0;
        for (ContainerReplicaOp op : pendingOps) {
            if (op.getOpType() != ContainerReplicaOp.PendingOpType.ADD) continue;
            ++pendingAdds;
        }
        if (pendingAdds >= vulnerableUnhealthy.size()) {
            LOG.debug("There are {} pending adds for container {}, while the number of UNHEALTHY replicas is {}.", new Object[]{pendingAdds, container.containerID(), vulnerableUnhealthy.size()});
            return 0;
        }
        Collections.shuffle(vulnerableUnhealthy);
        return this.replicateEachSource(replicaCount, vulnerableUnhealthy, pendingOps);
    }

    private int replicateEachSource(RatisContainerReplicaCount replicaCount, List<ContainerReplica> sources, List<ContainerReplicaOp> pendingOps) throws NotLeaderException, SCMException, CommandTargetOverloadedException {
        List<ContainerReplica> allReplicas = replicaCount.getReplicas();
        ContainerInfo container = replicaCount.getContainer();
        ReplicationManagerUtil.ExcludedAndUsedNodes excludedAndUsedNodes = ReplicationManagerUtil.getExcludedAndUsedNodes(container, allReplicas, Collections.emptySet(), pendingOps, this.replicationManager);
        CommandTargetOverloadedException firstException = null;
        int numCommandsSent = 0;
        for (ContainerReplica replica : sources) {
            int count;
            List<DatanodeDetails> target;
            block5: {
                target = ReplicationManagerUtil.getTargetDatanodes(this.placementPolicy, 1, excludedAndUsedNodes.getUsedNodes(), excludedAndUsedNodes.getExcludedNodes(), this.currentContainerSize, container);
                count = 0;
                try {
                    count = this.sendReplicationCommands(container, (List<DatanodeDetails>)ImmutableList.of((Object)replica.getDatanodeDetails()), target);
                }
                catch (CommandTargetOverloadedException e) {
                    LOG.info("Exception while replicating {} to target {} for container {}.", new Object[]{replica, target, container, e});
                    if (firstException != null) break block5;
                    firstException = e;
                }
            }
            if (count == 1) {
                excludedAndUsedNodes.getUsedNodes().add(target.get(0));
            }
            numCommandsSent += count;
        }
        if (firstException != null) {
            throw firstException;
        }
        return numCommandsSent;
    }

    private void removeUnhealthyReplicaIfPossible(ContainerInfo containerInfo, Set<ContainerReplica> replicas, List<ContainerReplicaOp> pendingOps) throws NotLeaderException {
        LOG.info("Finding an unhealthy replica to delete for container {} with replicas {} to unblock under replication handling.", (Object)containerInfo, replicas);
        int pendingDeletes = 0;
        for (ContainerReplicaOp op : pendingOps) {
            if (op.getOpType() != ContainerReplicaOp.PendingOpType.DELETE) continue;
            ++pendingDeletes;
        }
        ContainerReplica deleteCandidate = ReplicationManagerUtil.selectUnhealthyReplicaForDelete(containerInfo, replicas, pendingDeletes, dnd -> {
            try {
                return this.replicationManager.getNodeStatus((DatanodeDetails)dnd);
            }
            catch (NodeNotFoundException e) {
                LOG.warn("Exception while finding an unhealthy replica to delete for container {} with replicas {}.", new Object[]{containerInfo, replicas, e});
                return null;
            }
        });
        if (deleteCandidate != null) {
            this.replicationManager.sendDeleteCommand(containerInfo, deleteCandidate.getReplicaIndex(), deleteCandidate.getDatanodeDetails(), true);
        } else {
            LOG.info("Unable to find a replica to remove for container {} with replicas {}", (Object)containerInfo, replicas);
        }
    }

    private RatisContainerReplicaCount verifyUnderReplication(RatisContainerReplicaCount withUnhealthy, RatisContainerReplicaCount withoutUnhealthy) {
        if (withoutUnhealthy.isSufficientlyReplicated()) {
            LOG.info("The container {} state changed and it's not under replicated any more.", (Object)withoutUnhealthy.getContainer().containerID());
            return null;
        }
        if (withoutUnhealthy.isSufficientlyReplicated(true)) {
            LOG.info("Container {} with replicas {} will be sufficiently replicated after pending replicas are created.", (Object)withoutUnhealthy.getContainer().getContainerID(), withoutUnhealthy.getReplicas());
            return null;
        }
        if (withUnhealthy.getReplicas().isEmpty()) {
            LOG.warn("Container {} does not have any replicas and is unrecoverable.", (Object)withUnhealthy.getContainer());
            return null;
        }
        if (withUnhealthy.isSufficientlyReplicated(true) && withUnhealthy.getHealthyReplicaCount() == 0) {
            LOG.info("Container {} with only UNHEALTHY replicas [{}] will be sufficiently replicated after pending adds are created.", (Object)withUnhealthy.getContainer(), withUnhealthy.getReplicas());
            return null;
        }
        if (withoutUnhealthy.getHealthyReplicaCount() > 0) {
            return withoutUnhealthy;
        }
        return withUnhealthy;
    }

    private List<DatanodeDetails> getSources(RatisContainerReplicaCount replicaCount, List<ContainerReplicaOp> pendingOps) {
        HashSet<DatanodeDetails> pendingDeletion = new HashSet<DatanodeDetails>();
        for (ContainerReplicaOp op : pendingOps) {
            if (op.getOpType() != ContainerReplicaOp.PendingOpType.DELETE) continue;
            pendingDeletion.add(op.getTarget());
        }
        Predicate<ContainerReplica> predicate = replica -> replica.getState() == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSED;
        boolean hasClosedReplica = false;
        for (ContainerReplica replica2 : replicaCount.getReplicas()) {
            if (replica2.getState() != StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.CLOSED) continue;
            hasClosedReplica = true;
            break;
        }
        if (!hasClosedReplica || replicaCount.getContainer().getState() == HddsProtos.LifeCycleState.QUASI_CLOSED) {
            predicate = predicate.or(replica -> replica.getState() == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.QUASI_CLOSED);
        }
        if (replicaCount.getHealthyReplicaCount() == 0) {
            predicate = predicate.or(replica -> replica.getState() == StorageContainerDatanodeProtocolProtos.ContainerReplicaProto.State.UNHEALTHY);
        }
        List availableSources = replicaCount.getReplicas().stream().filter(predicate).filter(r -> {
            try {
                return this.replicationManager.getNodeStatus(r.getDatanodeDetails()).isHealthy();
            }
            catch (NodeNotFoundException e) {
                return false;
            }
        }).filter(r -> !pendingDeletion.contains(r.getDatanodeDetails())).collect(Collectors.toList());
        OptionalLong maxSequenceId = availableSources.stream().filter(r -> r.getSequenceId() != null).mapToLong(ContainerReplica::getSequenceId).max();
        Stream<Object> replicaStream = availableSources.stream();
        if (maxSequenceId.isPresent()) {
            replicaStream = replicaStream.filter(r -> r.getSequenceId() != null).filter(r -> r.getSequenceId().longValue() == maxSequenceId.getAsLong());
        }
        return replicaStream.map(ContainerReplica::getDatanodeDetails).collect(Collectors.toList());
    }

    private List<DatanodeDetails> getTargets(RatisContainerReplicaCount replicaCount, List<ContainerReplicaOp> pendingOps) throws IOException {
        LOG.debug("Need {} target datanodes for container {}. Current replicas: {}.", new Object[]{replicaCount.additionalReplicaNeeded(), replicaCount.getContainer().containerID(), replicaCount.getReplicas()});
        ReplicationManagerUtil.ExcludedAndUsedNodes excludedAndUsedNodes = ReplicationManagerUtil.getExcludedAndUsedNodes(replicaCount.getContainer(), replicaCount.getReplicas(), Collections.emptySet(), pendingOps, this.replicationManager);
        List<DatanodeDetails> excluded = excludedAndUsedNodes.getExcludedNodes();
        List<DatanodeDetails> used = excludedAndUsedNodes.getUsedNodes();
        LOG.debug("UsedList: {}, size {}. ExcludeList: {}, size: {}. ", new Object[]{used, used.size(), excluded, excluded.size()});
        return ReplicationManagerUtil.getTargetDatanodes(this.placementPolicy, replicaCount.additionalReplicaNeeded(), used, excluded, this.currentContainerSize, replicaCount.getContainer());
    }

    private int sendReplicationCommands(ContainerInfo containerInfo, List<DatanodeDetails> sources, List<DatanodeDetails> targets) throws CommandTargetOverloadedException, NotLeaderException {
        boolean push = this.replicationManager.getConfig().isPush();
        int commandsSent = 0;
        if (push) {
            for (DatanodeDetails target : targets) {
                this.replicationManager.sendThrottledReplicationCommand(containerInfo, sources, target, 0);
                ++commandsSent;
            }
        } else {
            for (DatanodeDetails target : targets) {
                ReplicateContainerCommand command = ReplicateContainerCommand.fromSources((long)containerInfo.getContainerID(), sources);
                this.replicationManager.sendDatanodeCommand((SCMCommand<?>)command, containerInfo, target);
                ++commandsSent;
            }
        }
        return commandsSent;
    }
}

