/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.security.symmetric;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.time.Duration;
import java.time.Instant;
import java.util.Objects;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.protocol.SecretKeyProtocol;
import org.apache.hadoop.hdds.security.exception.SCMSecretKeyException;
import org.apache.hadoop.hdds.security.symmetric.ManagedSecretKey;
import org.apache.hadoop.hdds.security.symmetric.SecretKeyConfig;
import org.apache.hadoop.hdds.security.symmetric.SecretKeySignerClient;
import org.apache.hadoop.hdds.utils.RetriableTask;
import org.apache.hadoop.io.retry.RetryPolicies;
import org.apache.hadoop.io.retry.RetryPolicy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultSecretKeySignerClient
implements SecretKeySignerClient {
    private static final Logger LOG = LoggerFactory.getLogger(DefaultSecretKeySignerClient.class);
    private final SecretKeyProtocol secretKeyProtocol;
    private final AtomicReference<ManagedSecretKey> cache = new AtomicReference();
    private final ThreadFactory threadFactory;
    private ScheduledExecutorService executorService;

    public DefaultSecretKeySignerClient(SecretKeyProtocol secretKeyProtocol, String threadNamePrefix) {
        this.secretKeyProtocol = secretKeyProtocol;
        this.threadFactory = new ThreadFactoryBuilder().setNameFormat(threadNamePrefix + "SecretKeyPoller").setDaemon(true).build();
    }

    @Override
    public ManagedSecretKey getCurrentSecretKey() {
        return Objects.requireNonNull(this.cache.get(), "SecretKey client must have been initialized already.");
    }

    @Override
    public void refetchSecretKey() {
        this.checkAndRefresh(Duration.ZERO);
    }

    @Override
    public void start(ConfigurationSource conf) throws IOException {
        ManagedSecretKey initialKey = this.loadInitialSecretKey();
        LOG.info("Initial secret key fetched from SCM: {}.", (Object)initialKey);
        this.cache.set(initialKey);
        this.scheduleSecretKeyPoller(conf, initialKey.getCreationTime());
    }

    private ManagedSecretKey loadInitialSecretKey() throws IOException {
        int maxRetries = 100;
        int backoffCircle = 10;
        int baseWaitTime = 1;
        RetryPolicy expBackoff = RetryPolicies.exponentialBackoffRetry((int)backoffCircle, (long)baseWaitTime, (TimeUnit)TimeUnit.SECONDS);
        RetryPolicy retryPolicy = (ex, retries, failovers, isIdempotent) -> {
            SCMSecretKeyException.ErrorCode errorCode;
            if (ex instanceof SCMSecretKeyException && (errorCode = ((SCMSecretKeyException)ex).getErrorCode()) == SCMSecretKeyException.ErrorCode.SECRET_KEY_NOT_INITIALIZED && retries < maxRetries) {
                return expBackoff.shouldRetry(ex, retries % backoffCircle, failovers, isIdempotent);
            }
            return RetryPolicy.RetryAction.FAIL;
        };
        RetriableTask task = new RetriableTask(retryPolicy, "getCurrentSecretKey", this.secretKeyProtocol::getCurrentSecretKey);
        try {
            return (ManagedSecretKey)task.call();
        }
        catch (IOException e) {
            throw e;
        }
        catch (Exception e) {
            throw new IllegalStateException("Unexpected exception getting current secret key", e);
        }
    }

    @Override
    public void stop() {
        if (this.executorService != null) {
            this.executorService.shutdown();
            try {
                if (this.executorService.awaitTermination(1L, TimeUnit.MINUTES)) {
                    this.executorService.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                LOG.error("Interrupted while shutting down executor service.", (Throwable)e);
                Thread.currentThread().interrupt();
            }
        }
    }

    private void scheduleSecretKeyPoller(ConfigurationSource conf, Instant initialCreation) {
        Duration rotateDuration = SecretKeyConfig.parseRotateDuration(conf);
        Instant nextRotate = initialCreation.plus(rotateDuration);
        this.executorService = Executors.newScheduledThreadPool(1, this.threadFactory);
        Duration interval = SecretKeyConfig.parseRotateCheckDuration(conf);
        Duration initialDelay = Duration.between(Instant.now(), nextRotate);
        LOG.info("Scheduling SecretKeyPoller with initial delay of {} and interval of {}", (Object)initialDelay, (Object)interval);
        this.executorService.scheduleAtFixedRate(() -> this.checkAndRefresh(rotateDuration), initialDelay.toMillis(), interval.toMillis(), TimeUnit.MILLISECONDS);
    }

    private synchronized void checkAndRefresh(Duration rotateDuration) {
        ManagedSecretKey current = this.cache.get();
        Instant nextRotate = current.getCreationTime().plus(rotateDuration);
        if (nextRotate.isBefore(Instant.now())) {
            try {
                ManagedSecretKey newKey = this.secretKeyProtocol.getCurrentSecretKey();
                if (!newKey.equals(current)) {
                    this.cache.set(newKey);
                    LOG.info("New secret key fetched from SCM: {}.", (Object)newKey);
                }
            }
            catch (IOException e) {
                throw new UncheckedIOException("Error fetching current key from SCM", e);
            }
        }
    }
}

