/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master.assignment;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import org.apache.hadoop.hbase.CatalogFamilyFormat;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.CellBuilderFactory;
import org.apache.hadoop.hbase.CellBuilderType;
import org.apache.hadoop.hbase.CellUtil;
import org.apache.hadoop.hbase.ClientMetaTableAccessor;
import org.apache.hadoop.hbase.DoNotRetryIOException;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.MetaTableAccessor;
import org.apache.hadoop.hbase.RegionLocations;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.AdvancedScanResultConsumer;
import org.apache.hadoop.hbase.client.AsyncTable;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.Mutation;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.RegionInfo;
import org.apache.hadoop.hbase.client.RegionInfoBuilder;
import org.apache.hadoop.hbase.client.RegionReplicaUtil;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.client.Table;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.hadoop.hbase.master.MasterFileSystem;
import org.apache.hadoop.hbase.master.MasterServices;
import org.apache.hadoop.hbase.master.RegionState;
import org.apache.hadoop.hbase.master.assignment.RegionStateNode;
import org.apache.hadoop.hbase.master.assignment.TransitRegionStateProcedure;
import org.apache.hadoop.hbase.master.region.MasterRegion;
import org.apache.hadoop.hbase.procedure2.util.StringUtils;
import org.apache.hadoop.hbase.replication.ReplicationBarrierFamilyFormat;
import org.apache.hadoop.hbase.shaded.org.apache.zookeeper.KeeperException;
import org.apache.hadoop.hbase.shaded.protobuf.ProtobufUtil;
import org.apache.hadoop.hbase.shaded.protobuf.generated.ClientProtos;
import org.apache.hadoop.hbase.shaded.protobuf.generated.MultiRowMutationProtos;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.hbase.util.FutureUtils;
import org.apache.hadoop.hbase.wal.WALSplitUtil;
import org.apache.hadoop.hbase.zookeeper.MetaTableLocator;
import org.apache.hbase.thirdparty.com.google.common.base.Preconditions;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class RegionStateStore {
    private static final Logger LOG = LoggerFactory.getLogger(RegionStateStore.class);
    private static final Logger METALOG = LoggerFactory.getLogger((String)"org.apache.hadoop.hbase.META");
    protected static final char META_REPLICA_ID_DELIMITER = '_';
    private final MasterServices master;
    private final MasterRegion masterRegion;

    public RegionStateStore(MasterServices master, MasterRegion masterRegion) {
        this.master = master;
        this.masterRegion = masterRegion;
    }

    public void visitMeta(final RegionStateVisitor visitor) throws IOException {
        MetaTableAccessor.fullScanRegions(this.master.getConnection(), new ClientMetaTableAccessor.Visitor(){
            final boolean isDebugEnabled = RegionStateStore.access$000().isDebugEnabled();

            @Override
            public boolean visit(Result r) throws IOException {
                if (r != null && !r.isEmpty()) {
                    long st = 0L;
                    if (LOG.isTraceEnabled()) {
                        st = EnvironmentEdgeManager.currentTime();
                    }
                    RegionStateStore.visitMetaEntry(visitor, r);
                    if (LOG.isTraceEnabled()) {
                        long et = EnvironmentEdgeManager.currentTime();
                        LOG.trace("[T] LOAD META PERF " + StringUtils.humanTimeDiff(et - st));
                    }
                } else if (this.isDebugEnabled) {
                    LOG.debug("NULL result from meta - ignoring but this is strange.");
                }
                return true;
            }
        });
    }

    public void visitMetaForRegion(String regionEncodedName, RegionStateVisitor visitor) throws IOException {
        Result result = MetaTableAccessor.scanByRegionEncodedName(this.master.getConnection(), regionEncodedName);
        if (result != null) {
            RegionStateStore.visitMetaEntry(visitor, result);
        }
    }

    public static void visitMetaEntry(RegionStateVisitor visitor, Result result) throws IOException {
        RegionLocations rl = CatalogFamilyFormat.getRegionLocations(result);
        if (rl == null) {
            return;
        }
        HRegionLocation[] locations = rl.getRegionLocations();
        if (locations == null) {
            return;
        }
        for (int i = 0; i < locations.length; ++i) {
            RegionInfo regionInfo;
            HRegionLocation hrl = locations[i];
            if (hrl == null || (regionInfo = hrl.getRegion()) == null) continue;
            int replicaId = regionInfo.getReplicaId();
            RegionState.State state = RegionStateStore.getRegionState(result, regionInfo);
            ServerName lastHost = hrl.getServerName();
            ServerName regionLocation = MetaTableAccessor.getTargetServerName(result, replicaId);
            long openSeqNum = hrl.getSeqNum();
            LOG.debug("Load hbase:meta entry region={}, regionState={}, lastHost={}, regionLocation={}, openSeqNum={}", new Object[]{regionInfo.getEncodedName(), state, lastHost, regionLocation, openSeqNum});
            visitor.visitRegionState(result, regionInfo, state, regionLocation, lastHost, openSeqNum);
        }
    }

    private Put generateUpdateRegionLocationPut(RegionStateNode regionStateNode) throws IOException {
        long time = EnvironmentEdgeManager.currentTime();
        long openSeqNum = regionStateNode.getState() == RegionState.State.OPEN ? regionStateNode.getOpenSeqNum() : -1L;
        RegionInfo regionInfo = regionStateNode.getRegionInfo();
        RegionState.State state = regionStateNode.getState();
        ServerName regionLocation = regionStateNode.getRegionLocation();
        TransitRegionStateProcedure rit = regionStateNode.getProcedure();
        long pid = rit != null ? rit.getProcId() : -1L;
        int replicaId = regionInfo.getReplicaId();
        Put put = new Put(CatalogFamilyFormat.getMetaKeyForRegion(regionInfo), time);
        MetaTableAccessor.addRegionInfo(put, regionInfo);
        StringBuilder info = new StringBuilder("pid=").append(pid).append(" updating hbase:meta row=").append(regionInfo.getEncodedName()).append(", regionState=").append((Object)state);
        if (openSeqNum >= 0L) {
            Preconditions.checkArgument(state == RegionState.State.OPEN && regionLocation != null, "Open region should be on a server");
            MetaTableAccessor.addLocation(put, regionLocation, openSeqNum, replicaId);
            if (regionInfo.getReplicaId() == 0 && this.hasGlobalReplicationScope(regionInfo.getTable())) {
                ReplicationBarrierFamilyFormat.addReplicationBarrier(put, openSeqNum);
                info.append(", repBarrier=").append(openSeqNum);
            }
            info.append(", openSeqNum=").append(openSeqNum);
            info.append(", regionLocation=").append(regionLocation);
        } else if (regionLocation != null) {
            put.add(CellBuilderFactory.create(CellBuilderType.SHALLOW_COPY).setRow(put.getRow()).setFamily(HConstants.CATALOG_FAMILY).setQualifier(CatalogFamilyFormat.getServerNameColumn(replicaId)).setTimestamp(put.getTimestamp()).setType(Cell.Type.Put).setValue(Bytes.toBytes(regionLocation.getServerName())).build());
            info.append(", regionLocation=").append(regionLocation);
        }
        put.add(CellBuilderFactory.create(CellBuilderType.SHALLOW_COPY).setRow(put.getRow()).setFamily(HConstants.CATALOG_FAMILY).setQualifier(RegionStateStore.getStateColumn(replicaId)).setTimestamp(put.getTimestamp()).setType(Cell.Type.Put).setValue(Bytes.toBytes(state.name())).build());
        LOG.info(info.toString());
        return put;
    }

    CompletableFuture<Void> updateRegionLocation(RegionStateNode regionStateNode) {
        Put put;
        try {
            put = this.generateUpdateRegionLocationPut(regionStateNode);
        }
        catch (IOException e) {
            return FutureUtils.failedFuture(e);
        }
        RegionInfo regionInfo = regionStateNode.getRegionInfo();
        RegionState.State state = regionStateNode.getState();
        CompletableFuture<Void> future = this.updateRegionLocation(regionInfo, state, put);
        if (regionInfo.isMetaRegion() && regionInfo.isFirst()) {
            assert (future.isDone());
            if (!future.isCompletedExceptionally()) {
                try {
                    this.mirrorMetaLocation(regionInfo, regionStateNode.getRegionLocation(), state);
                }
                catch (IOException e) {
                    return FutureUtils.failedFuture(e);
                }
            }
        }
        return future;
    }

    private void mirrorMetaLocation(RegionInfo regionInfo, ServerName serverName, RegionState.State state) throws IOException {
        try {
            MetaTableLocator.setMetaLocation(this.master.getZooKeeper(), serverName, regionInfo.getReplicaId(), state);
        }
        catch (KeeperException e) {
            throw new IOException(e);
        }
    }

    private void removeMirrorMetaLocation(int oldReplicaCount, int newReplicaCount) throws IOException {
        try {
            for (int i = newReplicaCount; i < oldReplicaCount; ++i) {
                MetaTableLocator.deleteMetaLocation(this.master.getZooKeeper(), i);
            }
        }
        catch (KeeperException e) {
            throw new IOException(e);
        }
    }

    private CompletableFuture<Void> updateRegionLocation(RegionInfo regionInfo, RegionState.State state, Put put) {
        CompletableFuture<Void> future;
        if (regionInfo.isMetaRegion()) {
            try {
                this.masterRegion.update(r -> r.put(put));
                future = CompletableFuture.completedFuture(null);
            }
            catch (Exception e2) {
                future = FutureUtils.failedFuture(e2);
            }
        } else {
            AsyncTable<AdvancedScanResultConsumer> table = this.master.getAsyncConnection().getTable(TableName.META_TABLE_NAME);
            future = table.put(put);
        }
        FutureUtils.addListener(future, (r, e) -> {
            if (e != null) {
                String msg = String.format("FAILED persisting region=%s state=%s", new Object[]{regionInfo.getShortNameToLog(), state});
                LOG.error(msg, e);
                this.master.abort(msg, (Throwable)e);
            }
        });
        return future;
    }

    private long getOpenSeqNumForParentRegion(RegionInfo region) throws IOException {
        MasterFileSystem fs = this.master.getMasterFileSystem();
        long maxSeqId = WALSplitUtil.getMaxRegionSequenceId(this.master.getConfiguration(), region, fs::getFileSystem, fs::getWALFileSystem);
        return maxSeqId > 0L ? maxSeqId + 1L : -1L;
    }

    private void multiMutate(RegionInfo ri, List<Mutation> mutations) throws IOException {
        RegionStateStore.debugLogMutations(mutations);
        byte[] row = Bytes.toBytes(RegionReplicaUtil.getRegionInfoForDefaultReplica(ri).getRegionNameAsString() + 44);
        MultiRowMutationProtos.MutateRowsRequest.Builder builder = MultiRowMutationProtos.MutateRowsRequest.newBuilder();
        for (Mutation mutation : mutations) {
            if (mutation instanceof Put) {
                builder.addMutationRequest(ProtobufUtil.toMutation(ClientProtos.MutationProto.MutationType.PUT, mutation));
                continue;
            }
            if (mutation instanceof Delete) {
                builder.addMutationRequest(ProtobufUtil.toMutation(ClientProtos.MutationProto.MutationType.DELETE, mutation));
                continue;
            }
            throw new DoNotRetryIOException("multi in MetaEditor doesn't support " + mutation.getClass().getName());
        }
        MultiRowMutationProtos.MutateRowsRequest request = builder.build();
        AsyncTable<AdvancedScanResultConsumer> table = this.master.getConnection().toAsyncConnection().getTable(TableName.META_TABLE_NAME);
        CompletableFuture future = table.coprocessorService(MultiRowMutationProtos.MultiRowMutationService::newStub, (stub, controller, done) -> stub.mutateRows(controller, request, done), row);
        FutureUtils.get(future);
    }

    private Table getMetaTable() throws IOException {
        return this.master.getConnection().getTable(TableName.META_TABLE_NAME);
    }

    private Result getRegionCatalogResult(RegionInfo region) throws IOException {
        Get get = new Get(CatalogFamilyFormat.getMetaKeyForRegion(region)).addFamily(HConstants.CATALOG_FAMILY);
        try (Table table = this.getMetaTable();){
            Result result = table.get(get);
            return result;
        }
    }

    private static Put addSequenceNum(Put p, long openSeqNum, int replicaId) throws IOException {
        return p.add(CellBuilderFactory.create(CellBuilderType.SHALLOW_COPY).setRow(p.getRow()).setFamily(HConstants.CATALOG_FAMILY).setQualifier(CatalogFamilyFormat.getSeqNumColumn(replicaId)).setTimestamp(p.getTimestamp()).setType(Cell.Type.Put).setValue(Bytes.toBytes(openSeqNum)).build());
    }

    public void splitRegion(RegionInfo parent, RegionInfo splitA, RegionInfo splitB, ServerName serverName, TableDescriptor htd) throws IOException {
        long parentOpenSeqNum = -1L;
        if (htd.hasGlobalReplicationScope()) {
            parentOpenSeqNum = this.getOpenSeqNumForParentRegion(parent);
        }
        long time = EnvironmentEdgeManager.currentTime();
        Put putParent = MetaTableAccessor.makePutFromRegionInfo(RegionInfoBuilder.newBuilder(parent).setOffline(true).setSplit(true).build(), time);
        MetaTableAccessor.addDaughtersToPut(putParent, splitA, splitB);
        Put putA = MetaTableAccessor.makePutFromRegionInfo(splitA, time);
        Put putB = MetaTableAccessor.makePutFromRegionInfo(splitB, time);
        if (parentOpenSeqNum > 0L) {
            ReplicationBarrierFamilyFormat.addReplicationBarrier(putParent, parentOpenSeqNum);
            ReplicationBarrierFamilyFormat.addReplicationParent(putA, Collections.singletonList(parent));
            ReplicationBarrierFamilyFormat.addReplicationParent(putB, Collections.singletonList(parent));
        }
        MetaTableAccessor.addRegionStateToPut(putA, 0, RegionState.State.CLOSED);
        MetaTableAccessor.addRegionStateToPut(putB, 0, RegionState.State.CLOSED);
        RegionStateStore.addSequenceNum(putA, 1L, splitA.getReplicaId());
        RegionStateStore.addSequenceNum(putB, 1L, splitB.getReplicaId());
        int regionReplication = this.getRegionReplication(htd);
        for (int i = 1; i < regionReplication; ++i) {
            MetaTableAccessor.addEmptyLocation(putA, i);
            MetaTableAccessor.addEmptyLocation(putB, i);
        }
        this.multiMutate(parent, Arrays.asList(putParent, putA, putB));
    }

    public void mergeRegions(RegionInfo child, RegionInfo[] parents, ServerName serverName, TableDescriptor htd) throws IOException {
        boolean globalScope = htd.hasGlobalReplicationScope();
        long time = Long.MAX_VALUE;
        ArrayList<Mutation> mutations = new ArrayList<Mutation>();
        ArrayList<RegionInfo> replicationParents = new ArrayList<RegionInfo>();
        for (RegionInfo ri : parents) {
            long seqNum = globalScope ? this.getOpenSeqNumForParentRegion(ri) : -1L;
            mutations.add(MetaTableAccessor.makeDeleteFromRegionInfo(ri, time));
            if (seqNum <= 0L) continue;
            mutations.add(ReplicationBarrierFamilyFormat.makePutForReplicationBarrier(ri, seqNum, time));
            replicationParents.add(ri);
        }
        Put putOfMerged = MetaTableAccessor.makePutFromRegionInfo(child, time);
        putOfMerged = RegionStateStore.addMergeRegions(putOfMerged, Arrays.asList(parents));
        MetaTableAccessor.addRegionStateToPut(putOfMerged, 0, RegionState.State.CLOSED);
        mutations.add(putOfMerged);
        if (serverName != null) {
            MetaTableAccessor.addLocation(putOfMerged, serverName, 1L, child.getReplicaId());
        }
        int regionReplication = this.getRegionReplication(htd);
        for (int i = 1; i < regionReplication; ++i) {
            MetaTableAccessor.addEmptyLocation(putOfMerged, i);
        }
        if (!replicationParents.isEmpty()) {
            ReplicationBarrierFamilyFormat.addReplicationParent(putOfMerged, replicationParents);
        }
        this.multiMutate(child, mutations);
    }

    public boolean hasMergeRegions(RegionInfo region) throws IOException {
        return CatalogFamilyFormat.hasMergeRegions(this.getRegionCatalogResult(region).rawCells());
    }

    public List<RegionInfo> getMergeRegions(RegionInfo region) throws IOException {
        return CatalogFamilyFormat.getMergeRegions(this.getRegionCatalogResult(region).rawCells());
    }

    public void deleteMergeQualifiers(RegionInfo mergeRegion) throws IOException {
        Cell[] cells = this.getRegionCatalogResult(mergeRegion).rawCells();
        if (cells == null || cells.length == 0) {
            return;
        }
        Delete delete = new Delete(mergeRegion.getRegionName());
        ArrayList<byte[]> qualifiers = new ArrayList<byte[]>();
        for (Cell cell : cells) {
            if (!CatalogFamilyFormat.isMergeQualifierPrefix(cell)) continue;
            byte[] qualifier = CellUtil.cloneQualifier(cell);
            qualifiers.add(qualifier);
            delete.addColumns(HConstants.CATALOG_FAMILY, qualifier, Long.MAX_VALUE);
        }
        if (qualifiers.isEmpty()) {
            LOG.info("No merged qualifiers for region " + mergeRegion.getRegionNameAsString() + " in meta table, they are cleaned up already, Skip.");
            return;
        }
        try (Table table = this.master.getConnection().getTable(TableName.META_TABLE_NAME);){
            table.delete(delete);
        }
        LOG.info("Deleted merge references in " + mergeRegion.getRegionNameAsString() + ", deleted qualifiers " + qualifiers.stream().map(Bytes::toStringBinary).collect(Collectors.joining(", ")));
    }

    static Put addMergeRegions(Put put, Collection<RegionInfo> mergeRegions) throws IOException {
        int limit = 10000;
        int max = mergeRegions.size();
        if (max > limit) {
            throw new RuntimeException("Can't merge " + max + " regions in one go; " + limit + " is upper-limit.");
        }
        int counter = 0;
        for (RegionInfo ri : mergeRegions) {
            String qualifier = String.format("merge%04d", counter++);
            put.add(CellBuilderFactory.create(CellBuilderType.SHALLOW_COPY).setRow(put.getRow()).setFamily(HConstants.CATALOG_FAMILY).setQualifier(Bytes.toBytes(qualifier)).setTimestamp(put.getTimestamp()).setType(Cell.Type.Put).setValue(RegionInfo.toByteArray(ri)).build());
        }
        return put;
    }

    public void deleteRegion(RegionInfo regionInfo) throws IOException {
        this.deleteRegions(Collections.singletonList(regionInfo));
    }

    public void deleteRegions(List<RegionInfo> regions) throws IOException {
        this.deleteRegions(regions, EnvironmentEdgeManager.currentTime());
    }

    private void deleteRegions(List<RegionInfo> regions, long ts) throws IOException {
        ArrayList<Delete> deletes = new ArrayList<Delete>(regions.size());
        for (RegionInfo hri : regions) {
            Delete e = new Delete(hri.getRegionName());
            e.addFamily(HConstants.CATALOG_FAMILY, ts);
            deletes.add(e);
        }
        try (Table table = this.getMetaTable();){
            RegionStateStore.debugLogMutations(deletes);
            table.delete(deletes);
        }
        LOG.info("Deleted {} regions from META", (Object)regions.size());
        LOG.debug("Deleted regions: {}", regions);
    }

    public void overwriteRegions(List<RegionInfo> regionInfos, int regionReplication) throws IOException {
        long now = EnvironmentEdgeManager.currentTime();
        this.deleteRegions(regionInfos, now);
        MetaTableAccessor.addRegionsToMeta(this.master.getConnection(), regionInfos, regionReplication, now + 1L);
        LOG.info("Overwritten " + regionInfos.size() + " regions to Meta");
        LOG.debug("Overwritten regions: {} ", regionInfos);
    }

    private Scan getScanForUpdateRegionReplicas(TableName tableName) {
        Scan scan = TableName.isMetaTableName(tableName) ? new Scan() : MetaTableAccessor.getScanForTableName(this.master.getConfiguration(), tableName);
        return scan.addColumn(HConstants.CATALOG_FAMILY, HConstants.REGIONINFO_QUALIFIER);
    }

    private List<Delete> deleteRegionReplicas(ResultScanner scanner, int oldReplicaCount, int newReplicaCount, long now) throws IOException {
        Result result;
        ArrayList<Delete> deletes = new ArrayList<Delete>();
        while ((result = scanner.next()) != null) {
            RegionInfo primaryRegionInfo = CatalogFamilyFormat.getRegionInfo(result);
            if (primaryRegionInfo == null || primaryRegionInfo.isSplit()) continue;
            Delete delete = new Delete(result.getRow());
            for (int i = newReplicaCount; i < oldReplicaCount; ++i) {
                delete.addColumns(HConstants.CATALOG_FAMILY, CatalogFamilyFormat.getServerColumn(i), now);
                delete.addColumns(HConstants.CATALOG_FAMILY, CatalogFamilyFormat.getSeqNumColumn(i), now);
                delete.addColumns(HConstants.CATALOG_FAMILY, CatalogFamilyFormat.getStartCodeColumn(i), now);
                delete.addColumns(HConstants.CATALOG_FAMILY, CatalogFamilyFormat.getServerNameColumn(i), now);
                delete.addColumns(HConstants.CATALOG_FAMILY, CatalogFamilyFormat.getRegionStateColumn(i), now);
            }
            deletes.add(delete);
        }
        return deletes;
    }

    public void removeRegionReplicas(TableName tableName, int oldReplicaCount, int newReplicaCount) throws IOException {
        Scan scan = this.getScanForUpdateRegionReplicas(tableName);
        long now = EnvironmentEdgeManager.currentTime();
        if (TableName.isMetaTableName(tableName)) {
            List<Delete> deletes;
            try (ResultScanner scanner = this.masterRegion.getScanner(scan);){
                deletes = this.deleteRegionReplicas(scanner, oldReplicaCount, newReplicaCount, now);
            }
            RegionStateStore.debugLogMutations(deletes);
            this.masterRegion.update(r -> {
                for (Delete d : deletes) {
                    r.delete(d);
                }
            });
            this.removeMirrorMetaLocation(oldReplicaCount, newReplicaCount);
        } else {
            try (Table metaTable = this.getMetaTable();
                 ResultScanner scanner = metaTable.getScanner(scan);){
                List<Delete> deletes = this.deleteRegionReplicas(scanner, oldReplicaCount, newReplicaCount, now);
                RegionStateStore.debugLogMutations(deletes);
                metaTable.delete(deletes);
            }
        }
    }

    private boolean hasGlobalReplicationScope(TableName tableName) throws IOException {
        return this.hasGlobalReplicationScope(this.getDescriptor(tableName));
    }

    private boolean hasGlobalReplicationScope(TableDescriptor htd) {
        return htd != null ? htd.hasGlobalReplicationScope() : false;
    }

    private int getRegionReplication(TableDescriptor htd) {
        return htd != null ? htd.getRegionReplication() : 1;
    }

    private TableDescriptor getDescriptor(TableName tableName) throws IOException {
        return this.master.getTableDescriptors().get(tableName);
    }

    public static RegionState.State getRegionState(Result r, RegionInfo regionInfo) {
        Cell cell = r.getColumnLatestCell(HConstants.CATALOG_FAMILY, RegionStateStore.getStateColumn(regionInfo.getReplicaId()));
        if (cell == null || cell.getValueLength() == 0) {
            return null;
        }
        String state = Bytes.toString(cell.getValueArray(), cell.getValueOffset(), cell.getValueLength());
        try {
            return RegionState.State.valueOf(state);
        }
        catch (IllegalArgumentException e) {
            LOG.warn("BAD value {} in hbase:meta info:state column for region {} , Consider using HBCK2 setRegionState ENCODED_REGION_NAME STATE", (Object)state, (Object)regionInfo.getEncodedName());
            return null;
        }
    }

    public static byte[] getStateColumn(int replicaId) {
        return replicaId == 0 ? HConstants.STATE_QUALIFIER : Bytes.toBytes("state_" + String.format("%04X", replicaId));
    }

    private static void debugLogMutations(List<? extends Mutation> mutations) throws IOException {
        if (!METALOG.isDebugEnabled()) {
            return;
        }
        for (Mutation mutation : mutations) {
            RegionStateStore.debugLogMutation(mutation);
        }
    }

    private static void debugLogMutation(Mutation p) throws IOException {
        METALOG.debug("{} {}", (Object)p.getClass().getSimpleName(), (Object)p.toJSON());
    }

    @FunctionalInterface
    public static interface RegionStateVisitor {
        public void visitRegionState(Result var1, RegionInfo var2, RegionState.State var3, ServerName var4, ServerName var5, long var6);
    }
}

