/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.replication.regionserver;

import com.google.errorprone.annotations.RestrictedApi;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Predicate;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.Server;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableDescriptors;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.regionserver.HRegionServer;
import org.apache.hadoop.hbase.regionserver.RegionServerCoprocessorHost;
import org.apache.hadoop.hbase.replication.ChainWALEntryFilter;
import org.apache.hadoop.hbase.replication.ClusterMarkingEntryFilter;
import org.apache.hadoop.hbase.replication.ReplicationEndpoint;
import org.apache.hadoop.hbase.replication.ReplicationException;
import org.apache.hadoop.hbase.replication.ReplicationGroupOffset;
import org.apache.hadoop.hbase.replication.ReplicationPeer;
import org.apache.hadoop.hbase.replication.ReplicationQueueData;
import org.apache.hadoop.hbase.replication.ReplicationQueueId;
import org.apache.hadoop.hbase.replication.ReplicationQueueStorage;
import org.apache.hadoop.hbase.replication.SystemTableWALEntryFilter;
import org.apache.hadoop.hbase.replication.WALEntryFilter;
import org.apache.hadoop.hbase.replication.regionserver.HBaseInterClusterReplicationEndpoint;
import org.apache.hadoop.hbase.replication.regionserver.MetricsSource;
import org.apache.hadoop.hbase.replication.regionserver.ReplicationSourceInterface;
import org.apache.hadoop.hbase.replication.regionserver.ReplicationSourceLogQueue;
import org.apache.hadoop.hbase.replication.regionserver.ReplicationSourceManager;
import org.apache.hadoop.hbase.replication.regionserver.ReplicationSourceShipper;
import org.apache.hadoop.hbase.replication.regionserver.ReplicationSourceWALReader;
import org.apache.hadoop.hbase.replication.regionserver.ReplicationStatus;
import org.apache.hadoop.hbase.replication.regionserver.ReplicationThrottler;
import org.apache.hadoop.hbase.replication.regionserver.SerialReplicationSourceWALReader;
import org.apache.hadoop.hbase.replication.regionserver.WALFileLengthProvider;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.OOMEChecker;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.hbase.util.Threads;
import org.apache.hadoop.hbase.wal.AbstractFSWALProvider;
import org.apache.hadoop.hbase.wal.WAL;
import org.apache.hbase.thirdparty.com.google.common.collect.ImmutableMap;
import org.apache.hbase.thirdparty.com.google.common.collect.Lists;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class ReplicationSource
implements ReplicationSourceInterface {
    private static final Logger LOG = LoggerFactory.getLogger(ReplicationSource.class);
    protected int queueSizePerGroup;
    protected ReplicationSourceLogQueue logQueue;
    protected ReplicationQueueStorage queueStorage;
    protected ReplicationPeer replicationPeer;
    protected Configuration conf;
    protected ReplicationSourceManager manager;
    protected Server server;
    private long sleepForRetries;
    protected FileSystem fs;
    private UUID clusterId;
    private AtomicLong totalReplicatedEdits = new AtomicLong(0L);
    protected ReplicationQueueId queueId;
    protected ImmutableMap<String, ReplicationGroupOffset> startOffsets;
    private int maxRetriesMultiplier;
    volatile boolean sourceRunning = false;
    private MetricsSource metrics;
    private volatile ReplicationEndpoint replicationEndpoint;
    private boolean abortOnError;
    private AtomicBoolean startupOngoing = new AtomicBoolean(false);
    private AtomicBoolean retryStartup = new AtomicBoolean(false);
    protected volatile WALEntryFilter walEntryFilter;
    private ReplicationThrottler throttler;
    private long defaultBandwidth;
    private long currentBandwidth;
    private WALFileLengthProvider walFileLengthProvider;
    protected final ConcurrentHashMap<String, ReplicationSourceShipper> workerThreads = new ConcurrentHashMap();
    public static final String WAIT_ON_ENDPOINT_SECONDS = "hbase.replication.wait.on.endpoint.seconds";
    public static final int DEFAULT_WAIT_ON_ENDPOINT_SECONDS = 30;
    private int waitOnEndpointSeconds = -1;
    private Thread initThread;
    private final Predicate<Path> filterInWALs;
    private final List<WALEntryFilter> baseFilterOutWALEntries;

    ReplicationSource() {
        this(p -> !AbstractFSWALProvider.isMetaFile(p), Lists.newArrayList((Object[])new WALEntryFilter[]{new SystemTableWALEntryFilter()}));
    }

    ReplicationSource(Predicate<Path> replicateWAL, List<WALEntryFilter> baseFilterOutWALEntries) {
        this.filterInWALs = replicateWAL;
        this.baseFilterOutWALEntries = Collections.unmodifiableList(baseFilterOutWALEntries);
    }

    @Override
    public void init(Configuration conf, FileSystem fs, ReplicationSourceManager manager, ReplicationQueueStorage queueStorage, ReplicationPeer replicationPeer, Server server, ReplicationQueueData queueData, UUID clusterId, WALFileLengthProvider walFileLengthProvider, MetricsSource metrics) throws IOException {
        this.server = server;
        this.conf = HBaseConfiguration.create((Configuration)conf);
        this.waitOnEndpointSeconds = this.conf.getInt(WAIT_ON_ENDPOINT_SECONDS, 30);
        this.decorateConf();
        this.sleepForRetries = this.conf.getLong("replication.source.sleepforretries", 1000L);
        this.maxRetriesMultiplier = this.conf.getInt("replication.source.maxretriesmultiplier", 300);
        this.queueSizePerGroup = this.conf.getInt("hbase.regionserver.maxlogs", 32);
        this.logQueue = new ReplicationSourceLogQueue(conf, metrics, this);
        this.queueStorage = queueStorage;
        this.replicationPeer = replicationPeer;
        this.manager = manager;
        this.fs = fs;
        this.metrics = metrics;
        this.clusterId = clusterId;
        this.queueId = queueData.getId();
        this.startOffsets = queueData.getOffsets();
        this.defaultBandwidth = this.conf.getLong("replication.source.per.peer.node.bandwidth", 0L);
        this.currentBandwidth = this.getCurrentBandwidth();
        this.throttler = new ReplicationThrottler((double)this.currentBandwidth / 10.0);
        this.walFileLengthProvider = walFileLengthProvider;
        this.abortOnError = this.conf.getBoolean("replication.source.regionserver.abort", true);
        LOG.info("queueId={}, ReplicationSource: {}, currentBandwidth={}", new Object[]{this.queueId, replicationPeer.getId(), this.currentBandwidth});
    }

    private void decorateConf() {
        String replicationCodec = this.conf.get("hbase.replication.rpc.codec");
        if (StringUtils.isNotEmpty((CharSequence)replicationCodec)) {
            this.conf.set("hbase.client.rpc.codec", replicationCodec);
        }
    }

    @Override
    public void enqueueLog(Path wal) {
        if (!this.filterInWALs.test(wal)) {
            LOG.trace("NOT replicating {}", (Object)wal);
            return;
        }
        String walGroupId = AbstractFSWALProvider.getWALPrefixFromWALName(wal.getName());
        boolean queueExists = this.logQueue.enqueueLog(wal, walGroupId);
        if (!queueExists && this.isSourceActive() && this.walEntryFilter != null) {
            this.tryStartNewShipper(walGroupId);
        }
        if (LOG.isTraceEnabled()) {
            LOG.trace("{} Added wal {} to queue of source {}.", new Object[]{this.logPeerId(), walGroupId, this.queueId});
        }
    }

    @RestrictedApi(explanation="Should only be called in tests", link="", allowedOnPath=".*/src/test/.*")
    public Map<String, PriorityBlockingQueue<Path>> getQueues() {
        return this.logQueue.getQueues();
    }

    @Override
    public void addHFileRefs(TableName tableName, byte[] family, List<Pair<Path, Path>> pairs) throws ReplicationException {
        String peerId = this.replicationPeer.getId();
        if (this.replicationPeer.getPeerConfig().needToReplicate(tableName, family)) {
            this.queueStorage.addHFileRefs(peerId, pairs);
            this.metrics.incrSizeOfHFileRefsQueue(pairs.size());
        } else {
            LOG.debug("HFiles will not be replicated belonging to the table {} family {} to peer id {}", new Object[]{tableName, Bytes.toString((byte[])family), peerId});
        }
    }

    private ReplicationEndpoint createReplicationEndpoint() throws InstantiationException, IllegalAccessException, ClassNotFoundException, IOException {
        ReplicationEndpoint newReplicationEndPoint;
        ReplicationEndpoint replicationEndpoint;
        String replicationEndpointImpl;
        RegionServerCoprocessorHost rsServerHost = null;
        if (this.server instanceof HRegionServer) {
            rsServerHost = ((HRegionServer)this.server).getRegionServerCoprocessorHost();
        }
        if ((replicationEndpointImpl = this.replicationPeer.getPeerConfig().getReplicationEndpointImpl()) == null) {
            replicationEndpoint = new HBaseInterClusterReplicationEndpoint();
        } else {
            try {
                replicationEndpoint = Class.forName(replicationEndpointImpl).asSubclass(ReplicationEndpoint.class).getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
            }
            catch (NoSuchMethodException | InvocationTargetException e) {
                throw new IllegalArgumentException(e);
            }
        }
        if (rsServerHost != null && (newReplicationEndPoint = rsServerHost.postCreateReplicationEndPoint(replicationEndpoint)) != null) {
            replicationEndpoint = newReplicationEndPoint;
        }
        return replicationEndpoint;
    }

    private void initAndStartReplicationEndpoint(ReplicationEndpoint replicationEndpoint) throws IOException, TimeoutException {
        TableDescriptors tableDescriptors = null;
        if (this.server instanceof HRegionServer) {
            tableDescriptors = ((HRegionServer)this.server).getTableDescriptors();
        }
        replicationEndpoint.init(new ReplicationEndpoint.Context(this.server, this.conf, this.replicationPeer.getConfiguration(), this.fs, this.replicationPeer.getId(), this.clusterId, this.replicationPeer, this.metrics, tableDescriptors, this.server));
        replicationEndpoint.start();
        replicationEndpoint.awaitRunning(this.waitOnEndpointSeconds, TimeUnit.SECONDS);
    }

    private void initializeWALEntryFilter(UUID peerClusterId) {
        ArrayList<WALEntryFilter> filters = new ArrayList<WALEntryFilter>(this.baseFilterOutWALEntries);
        WALEntryFilter filterFromEndpoint = this.replicationEndpoint.getWALEntryfilter();
        if (filterFromEndpoint != null) {
            filters.add(filterFromEndpoint);
        }
        filters.add(new ClusterMarkingEntryFilter(this.clusterId, peerClusterId, this.replicationEndpoint));
        this.walEntryFilter = new ChainWALEntryFilter(filters);
    }

    private long getStartOffset(String walGroupId) {
        ReplicationGroupOffset startOffset = (ReplicationGroupOffset)this.startOffsets.get((Object)walGroupId);
        if (startOffset == null || startOffset == ReplicationGroupOffset.BEGIN) {
            return 0L;
        }
        Path first = this.logQueue.getQueue(walGroupId).peek();
        if (!startOffset.getWal().equals(first.getName())) {
            return 0L;
        }
        if (startOffset.getOffset() < 0L) {
            LOG.warn("Should have already replicated wal {}, return start offset as 0", (Object)startOffset.getWal());
            return 0L;
        }
        return startOffset.getOffset();
    }

    protected final ReplicationSourceShipper createNewShipper(String walGroupId) {
        ReplicationSourceWALReader walReader = this.createNewWALReader(walGroupId, this.getStartOffset(walGroupId));
        ReplicationSourceShipper worker = this.createNewShipper(walGroupId, walReader);
        Threads.setDaemonThreadRunning((Thread)walReader, (String)(Thread.currentThread().getName() + ".replicationSource.wal-reader." + walGroupId + "," + this.queueId), this::retryRefreshing);
        return worker;
    }

    protected final void startShipper(ReplicationSourceShipper worker) {
        worker.startup(this::retryRefreshing);
    }

    private void tryStartNewShipper(String walGroupId) {
        this.workerThreads.compute(walGroupId, (key, value) -> {
            if (value != null) {
                LOG.debug("{} preempted start of shipping worker walGroupId={}", (Object)this.logPeerId(), (Object)walGroupId);
                return value;
            }
            LOG.debug("{} starting shipping worker for walGroupId={}", (Object)this.logPeerId(), (Object)walGroupId);
            ReplicationSourceShipper worker = this.createNewShipper(walGroupId);
            this.startShipper(worker);
            return worker;
        });
    }

    @Override
    public Map<String, ReplicationStatus> getWalGroupStatus() {
        TreeMap<String, ReplicationStatus> sourceReplicationStatus = new TreeMap<String, ReplicationStatus>();
        for (Map.Entry<String, ReplicationSourceShipper> walGroupShipper : this.workerThreads.entrySet()) {
            String walGroupId = walGroupShipper.getKey();
            ReplicationSourceShipper shipper = walGroupShipper.getValue();
            long ageOfLastShippedOp = this.metrics.getAgeOfLastShippedOp(walGroupId);
            int queueSize = this.logQueue.getQueueSize(walGroupId);
            long replicationDelay = this.metrics.getReplicationDelay();
            Path currentPath = shipper.getCurrentPath();
            long fileSize = -1L;
            if (currentPath != null) {
                try {
                    fileSize = this.getFileSize(currentPath);
                }
                catch (IOException e) {
                    LOG.warn("Ignore the exception as the file size of HLog only affects the web ui", (Throwable)e);
                }
            } else {
                currentPath = new Path("NO_LOGS_IN_QUEUE");
                LOG.warn("{} No replication ongoing, waiting for new log", (Object)this.logPeerId());
            }
            ReplicationStatus.ReplicationStatusBuilder statusBuilder = ReplicationStatus.newBuilder();
            statusBuilder.withPeerId(this.getPeerId()).withQueueSize(queueSize).withWalGroup(walGroupId).withCurrentPath(currentPath).withCurrentPosition(shipper.getCurrentPosition()).withFileSize(fileSize).withAgeOfLastShippedOp(ageOfLastShippedOp).withReplicationDelay(replicationDelay);
            sourceReplicationStatus.put(this.getPeerId() + "=>" + walGroupId, statusBuilder.build());
        }
        return sourceReplicationStatus;
    }

    private long getFileSize(Path currentPath) throws IOException {
        long fileSize;
        try {
            fileSize = this.fs.getContentSummary(currentPath).getLength();
        }
        catch (FileNotFoundException e) {
            Path archivedLogPath = AbstractFSWALProvider.findArchivedLog(currentPath, this.conf);
            if (archivedLogPath == null) {
                throw new FileNotFoundException("Couldn't find path: " + currentPath);
            }
            fileSize = this.fs.getContentSummary(archivedLogPath).getLength();
        }
        return fileSize;
    }

    protected ReplicationSourceShipper createNewShipper(String walGroupId, ReplicationSourceWALReader walReader) {
        return new ReplicationSourceShipper(this.conf, walGroupId, this, walReader);
    }

    private ReplicationSourceWALReader createNewWALReader(String walGroupId, long startPosition) {
        return this.replicationPeer.getPeerConfig().isSerial() ? new SerialReplicationSourceWALReader(this.fs, this.conf, this.logQueue, startPosition, this.walEntryFilter, this, walGroupId) : new ReplicationSourceWALReader(this.fs, this.conf, this.logQueue, startPosition, this.walEntryFilter, this, walGroupId);
    }

    WALEntryFilter getWalEntryFilter() {
        return this.walEntryFilter;
    }

    private void checkError(Thread t, Throwable error) {
        OOMEChecker.exitIfOOME(error, this.getClass().getSimpleName());
        LOG.error("Unexpected exception in {} currentPath={}", new Object[]{t.getName(), this.getCurrentPath(), error});
        if (this.abortOnError) {
            this.server.abort("Unexpected exception in " + t.getName(), error);
        }
    }

    private void retryRefreshing(Thread t, Throwable error) {
        this.checkError(t, error);
        while (true) {
            if (this.server.isAborted() || this.server.isStopped() || this.server.isStopping()) {
                LOG.warn("Server is shutting down, give up refreshing source for peer {}", (Object)this.getPeerId());
                return;
            }
            try {
                LOG.info("Refreshing replication sources now due to previous error on thread: {}", (Object)t.getName());
                this.manager.refreshSources(this.getPeerId());
            }
            catch (Exception e) {
                LOG.error("Replication sources refresh failed.", (Throwable)e);
                this.sleepForRetries("Sleeping before try refreshing sources again", this.maxRetriesMultiplier);
                continue;
            }
            break;
        }
    }

    @Override
    public ReplicationEndpoint getReplicationEndpoint() {
        return this.replicationEndpoint;
    }

    @Override
    public ReplicationSourceManager getSourceManager() {
        return this.manager;
    }

    @Override
    public void tryThrottle(int batchSize) throws InterruptedException {
        long sleepTicks;
        this.checkBandwidthChangeAndResetThrottler();
        if (this.throttler.isEnabled() && (sleepTicks = this.throttler.getNextSleepInterval(batchSize)) > 0L) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("{} To sleep {}ms for throttling control", (Object)this.logPeerId(), (Object)sleepTicks);
            }
            Thread.sleep(sleepTicks);
            this.throttler.resetStartTick();
        }
    }

    private void checkBandwidthChangeAndResetThrottler() {
        long peerBandwidth = this.getCurrentBandwidth();
        if (peerBandwidth != this.currentBandwidth) {
            this.currentBandwidth = peerBandwidth;
            this.throttler.setBandwidth((double)this.currentBandwidth / 10.0);
            LOG.info("ReplicationSource : {} bandwidth throttling changed, currentBandWidth={}", (Object)this.replicationPeer.getId(), (Object)this.currentBandwidth);
        }
    }

    private long getCurrentBandwidth() {
        long peerBandwidth = this.replicationPeer.getPeerBandwidth();
        return peerBandwidth != 0L ? peerBandwidth : this.defaultBandwidth;
    }

    private boolean sleepForRetries(String msg, int sleepMultiplier) {
        try {
            if (LOG.isTraceEnabled()) {
                LOG.trace("{} {}, sleeping {} times {}", new Object[]{this.logPeerId(), msg, this.sleepForRetries, sleepMultiplier});
            }
            Thread.sleep(this.sleepForRetries * (long)sleepMultiplier);
        }
        catch (InterruptedException e) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("{} Interrupted while sleeping between retries", (Object)this.logPeerId());
            }
            Thread.currentThread().interrupt();
        }
        return sleepMultiplier < this.maxRetriesMultiplier;
    }

    private void initialize() {
        UUID peerClusterId;
        int sleepMultiplier = 1;
        while (this.isSourceActive()) {
            ReplicationEndpoint replicationEndpoint;
            try {
                replicationEndpoint = this.createReplicationEndpoint();
            }
            catch (Exception e) {
                LOG.warn("{} error creating ReplicationEndpoint, retry", (Object)this.logPeerId(), (Object)e);
                if (!this.sleepForRetries("Error creating ReplicationEndpoint", sleepMultiplier)) continue;
                ++sleepMultiplier;
                continue;
            }
            try {
                this.initAndStartReplicationEndpoint(replicationEndpoint);
                this.replicationEndpoint = replicationEndpoint;
                break;
            }
            catch (Exception e) {
                LOG.warn("{} Error starting ReplicationEndpoint, retry", (Object)this.logPeerId(), (Object)e);
                replicationEndpoint.stop();
                if (this.sleepForRetries("Error starting ReplicationEndpoint", sleepMultiplier)) {
                    ++sleepMultiplier;
                    continue;
                }
                this.retryStartup.set(!this.abortOnError);
                this.setSourceStartupStatus(false);
                throw new RuntimeException("Exhausted retries to start replication endpoint.");
            }
        }
        if (!this.isSourceActive()) {
            this.setSourceStartupStatus(false);
            return;
        }
        sleepMultiplier = 1;
        while (true) {
            peerClusterId = this.replicationEndpoint.getPeerUUID();
            if (!this.isSourceActive() || peerClusterId != null) break;
            if (LOG.isDebugEnabled()) {
                LOG.debug("{} Could not connect to Peer ZK. Sleeping for {} millis", (Object)this.logPeerId(), (Object)(this.sleepForRetries * (long)sleepMultiplier));
            }
            if (!this.sleepForRetries("Cannot contact the peer's zk ensemble", sleepMultiplier)) continue;
            ++sleepMultiplier;
        }
        if (!this.isSourceActive()) {
            this.setSourceStartupStatus(false);
            return;
        }
        LOG.info("{} queueId={} (queues={}) is replicating from cluster={} to cluster={}", new Object[]{this.logPeerId(), this.queueId, this.logQueue.getNumQueues(), this.clusterId, peerClusterId});
        this.initializeWALEntryFilter(peerClusterId);
        this.startShippers();
        this.setSourceStartupStatus(false);
    }

    protected void startShippers() {
        for (String walGroupId : this.logQueue.getQueues().keySet()) {
            this.tryStartNewShipper(walGroupId);
        }
    }

    private synchronized void setSourceStartupStatus(boolean initializing) {
        this.startupOngoing.set(initializing);
        if (initializing) {
            this.metrics.incrSourceInitializing();
        } else {
            this.metrics.decrSourceInitializing();
        }
    }

    @Override
    public ReplicationSourceInterface startup() {
        if (this.sourceRunning) {
            return this;
        }
        this.sourceRunning = true;
        this.setSourceStartupStatus(true);
        this.initThread = new Thread(this::initialize);
        Threads.setDaemonThreadRunning((Thread)this.initThread, (String)(Thread.currentThread().getName() + ".replicationSource," + this.queueId), (t, e) -> {
            this.sourceRunning = false;
            this.checkError(t, e);
            this.retryStartup.set(!this.abortOnError);
            do {
                if (!this.retryStartup.get()) continue;
                this.sourceRunning = true;
                this.setSourceStartupStatus(true);
                this.retryStartup.set(false);
                try {
                    this.initialize();
                }
                catch (Throwable error) {
                    this.setSourceStartupStatus(false);
                    this.checkError(t, error);
                    this.retryStartup.set(!this.abortOnError);
                }
            } while ((this.startupOngoing.get() || this.retryStartup.get()) && !this.abortOnError);
        });
        return this;
    }

    @Override
    public void terminate(String reason) {
        this.terminate(reason, null);
    }

    @Override
    public void terminate(String reason, Exception cause) {
        this.terminate(reason, cause, true);
    }

    @Override
    public void terminate(String reason, Exception cause, boolean clearMetrics) {
        this.terminate(reason, cause, clearMetrics, true);
    }

    private void terminate(String reason, Exception cause, boolean clearMetrics, boolean join) {
        if (cause == null) {
            LOG.info("{} Closing source {} because: {}", new Object[]{this.logPeerId(), this.queueId, reason});
        } else {
            LOG.error(String.format("%s Closing source %s because an error occurred: %s", this.logPeerId(), this.queueId, reason), (Throwable)cause);
        }
        this.sourceRunning = false;
        if (this.initThread != null && Thread.currentThread() != this.initThread) {
            this.initThread.interrupt();
            Threads.shutdown((Thread)this.initThread, (long)this.sleepForRetries);
        }
        Collection<ReplicationSourceShipper> workers = this.workerThreads.values();
        for (ReplicationSourceShipper worker : workers) {
            worker.stopWorker();
            worker.entryReader.setReaderRunning(false);
        }
        if (this.replicationEndpoint != null) {
            this.replicationEndpoint.stop();
        }
        for (ReplicationSourceShipper worker : workers) {
            if (worker.isAlive() || worker.entryReader.isAlive()) {
                try {
                    Thread.sleep(this.sleepForRetries);
                }
                catch (InterruptedException e) {
                    LOG.info("{} Interrupted while waiting {} to stop", (Object)this.logPeerId(), (Object)worker.getName());
                    Thread.currentThread().interrupt();
                }
                if (worker.isAlive()) {
                    worker.interrupt();
                }
                if (worker.entryReader.isAlive()) {
                    worker.entryReader.interrupt();
                }
            }
            if (this.server.isAborted() || this.server.isStopped()) continue;
            worker.clearWALEntryBatch();
        }
        if (join) {
            for (ReplicationSourceShipper worker : workers) {
                Threads.shutdown((Thread)worker, (long)this.sleepForRetries);
                LOG.info("{} ReplicationSourceWorker {} terminated", (Object)this.logPeerId(), (Object)worker.getName());
            }
            if (this.replicationEndpoint != null) {
                try {
                    this.replicationEndpoint.awaitTerminated(this.sleepForRetries * (long)this.maxRetriesMultiplier, TimeUnit.MILLISECONDS);
                }
                catch (TimeoutException te) {
                    LOG.warn("{} Got exception while waiting for endpoint to shutdown for replication source : {}", new Object[]{this.logPeerId(), this.queueId, te});
                }
            }
        }
        if (this.metrics != null) {
            if (clearMetrics) {
                this.metrics.clear();
            } else {
                this.metrics.terminate();
            }
        }
    }

    @Override
    public ReplicationQueueId getQueueId() {
        return this.queueId;
    }

    @Override
    public Path getCurrentPath() {
        for (ReplicationSourceShipper worker : this.workerThreads.values()) {
            if (worker.getCurrentPath() == null) continue;
            return worker.getCurrentPath();
        }
        return null;
    }

    @Override
    public boolean isSourceActive() {
        return !this.server.isStopped() && this.sourceRunning;
    }

    public boolean isWorkerRunning() {
        for (ReplicationSourceShipper worker : this.workerThreads.values()) {
            if (!worker.isActive()) continue;
            return worker.isActive();
        }
        return false;
    }

    @Override
    public String getStats() {
        StringBuilder sb = new StringBuilder();
        sb.append("Total replicated edits: ").append(this.totalReplicatedEdits).append(", current progress: \n");
        for (Map.Entry<String, ReplicationSourceShipper> entry : this.workerThreads.entrySet()) {
            String walGroupId = entry.getKey();
            ReplicationSourceShipper worker = entry.getValue();
            long position = worker.getCurrentPosition();
            Path currentPath = worker.getCurrentPath();
            sb.append("walGroup [").append(walGroupId).append("]: ");
            if (currentPath != null) {
                sb.append("currently replicating from: ").append(currentPath).append(" at position: ").append(position).append("\n");
                continue;
            }
            sb.append("no replication ongoing, waiting for new log").append("\n");
        }
        return sb.toString();
    }

    @Override
    public MetricsSource getSourceMetrics() {
        return this.metrics;
    }

    @Override
    public void postShipEdits(List<WAL.Entry> entries, long batchSize) {
        if (this.throttler.isEnabled()) {
            this.throttler.addPushSize(batchSize);
        }
        this.totalReplicatedEdits.addAndGet(entries.size());
        this.manager.releaseBufferQuota(batchSize);
    }

    @Override
    public WALFileLengthProvider getWALFileLengthProvider() {
        return this.walFileLengthProvider;
    }

    @Override
    public ServerName getServerWALsBelongTo() {
        return this.queueId.getServerWALsBelongTo();
    }

    @Override
    public ReplicationPeer getPeer() {
        return this.replicationPeer;
    }

    Server getServer() {
        return this.server;
    }

    @Override
    public ReplicationQueueStorage getReplicationQueueStorage() {
        return this.queueStorage;
    }

    void removeWorker(ReplicationSourceShipper worker) {
        this.workerThreads.remove(worker.walGroupId, worker);
    }

    public String logPeerId() {
        return "peerId=" + this.getPeerId() + ",";
    }

    public long getTotalReplicatedEdits() {
        return this.totalReplicatedEdits.get();
    }
}

