/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.regionserver.storefiletracker;

import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;
import java.util.concurrent.ForkJoinPool;
import java.util.regex.Pattern;
import java.util.zip.CRC32;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.DoNotRetryIOException;
import org.apache.hadoop.hbase.regionserver.StoreContext;
import org.apache.hadoop.hbase.shaded.protobuf.generated.StoreFileTrackerProtos;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hbase.thirdparty.com.google.common.base.Splitter;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
class StoreFileListFile {
    private static final Logger LOG = LoggerFactory.getLogger(StoreFileListFile.class);
    static final String TRACK_FILE_DIR = ".filelist";
    private static final String TRACK_FILE_PREFIX = "f1";
    private static final String TRACK_FILE_ROTATE_PREFIX = "f2";
    private static final char TRACK_FILE_SEPARATOR = '.';
    static final Pattern TRACK_FILE_PATTERN = Pattern.compile("^f(1|2)\\.\\d+$");
    private static final int MAX_FILE_SIZE = 0x1000000;
    private final StoreContext ctx;
    private final Path trackFileDir;
    private final Path[] trackFiles = new Path[2];
    private long prevTimestamp = -1L;
    private int nextTrackFile = -1;

    StoreFileListFile(StoreContext ctx) {
        this.ctx = ctx;
        this.trackFileDir = new Path(ctx.getFamilyStoreDirectoryPath(), TRACK_FILE_DIR);
    }

    static StoreFileTrackerProtos.StoreFileList load(FileSystem fs, Path path) throws IOException {
        int expectedChecksum;
        byte[] data;
        try (FSDataInputStream in = fs.open(path);){
            int length = in.readInt();
            if (length <= 0 || length > 0x1000000) {
                throw new IOException("Invalid file length " + length + ", either less than 0 or greater then max allowed size " + 0x1000000);
            }
            data = new byte[length];
            in.readFully(data);
            expectedChecksum = in.readInt();
        }
        CRC32 crc32 = new CRC32();
        crc32.update(data);
        int calculatedChecksum = (int)crc32.getValue();
        if (expectedChecksum != calculatedChecksum) {
            throw new IOException("Checksum mismatch, expected " + expectedChecksum + ", actual " + calculatedChecksum);
        }
        return StoreFileTrackerProtos.StoreFileList.parseFrom((byte[])data);
    }

    StoreFileTrackerProtos.StoreFileList load(Path path) throws IOException {
        FileSystem fs = this.ctx.getRegionFileSystem().getFileSystem();
        return StoreFileListFile.load(fs, path);
    }

    private int select(StoreFileTrackerProtos.StoreFileList[] lists) {
        if (lists[0] == null) {
            return 1;
        }
        if (lists[1] == null) {
            return 0;
        }
        return lists[0].getTimestamp() >= lists[1].getTimestamp() ? 0 : 1;
    }

    private NavigableMap<Long, List<Path>> listFiles() throws IOException {
        FileStatus[] statuses;
        FileSystem fs = this.ctx.getRegionFileSystem().getFileSystem();
        try {
            statuses = fs.listStatus(this.trackFileDir);
        }
        catch (FileNotFoundException e) {
            LOG.debug("Track file directory {} does not exist", (Object)this.trackFileDir, (Object)e);
            return Collections.emptyNavigableMap();
        }
        if (statuses == null || statuses.length == 0) {
            return Collections.emptyNavigableMap();
        }
        TreeMap<Long, List<Path>> map = new TreeMap<Long, List<Path>>((l1, l2) -> l2.compareTo((Long)l1));
        for (FileStatus status : statuses) {
            Path file = status.getPath();
            if (!status.isFile()) {
                LOG.warn("Found invalid track file {}, which is not a file", (Object)file);
                continue;
            }
            if (!TRACK_FILE_PATTERN.matcher(file.getName()).matches()) {
                LOG.warn("Found invalid track file {}, skip", (Object)file);
                continue;
            }
            List parts = Splitter.on((char)'.').splitToList((CharSequence)file.getName());
            map.computeIfAbsent(Long.parseLong((String)parts.get(1)), k -> new ArrayList()).add(file);
        }
        return map;
    }

    private void initializeTrackFiles(long seqId) {
        this.trackFiles[0] = new Path(this.trackFileDir, "f1." + seqId);
        this.trackFiles[1] = new Path(this.trackFileDir, "f2." + seqId);
        LOG.info("Initialized track files: {}, {}", (Object)this.trackFiles[0], (Object)this.trackFiles[1]);
    }

    private void cleanUpTrackFiles(long loadedSeqId, NavigableMap<Long, List<Path>> seqId2TrackFiles) {
        LOG.info("Cleanup track file with sequence id < {}", (Object)loadedSeqId);
        FileSystem fs = this.ctx.getRegionFileSystem().getFileSystem();
        NavigableMap<Long, List<Path>> toDelete = loadedSeqId >= 0L ? seqId2TrackFiles.tailMap(loadedSeqId, false) : seqId2TrackFiles;
        toDelete.values().stream().flatMap(l -> l.stream()).forEach(file -> ForkJoinPool.commonPool().execute(() -> {
            LOG.info("Deleting track file {}", file);
            try {
                fs.delete(file, false);
            }
            catch (IOException e) {
                LOG.warn("failed to delete unused track file {}", file, (Object)e);
            }
        }));
    }

    StoreFileTrackerProtos.StoreFileList load(boolean readOnly) throws IOException {
        NavigableMap<Long, List<Path>> seqId2TrackFiles = this.listFiles();
        long seqId = -1L;
        StoreFileTrackerProtos.StoreFileList[] lists = new StoreFileTrackerProtos.StoreFileList[2];
        for (Map.Entry entry : seqId2TrackFiles.entrySet()) {
            List files = (List)entry.getValue();
            if (files.size() > 2) {
                throw new DoNotRetryIOException("Should only have at most 2 track files for sequence id " + entry.getKey() + ", but got " + files.size() + " files: " + files);
            }
            boolean loaded = false;
            for (int i = 0; i < files.size(); ++i) {
                try {
                    lists[i] = this.load((Path)files.get(i));
                    loaded = true;
                    continue;
                }
                catch (EOFException e) {
                    LOG.debug("EOF loading track file {}, ignoring the exception", (Object)this.trackFiles[i], (Object)e);
                }
            }
            if (!loaded) continue;
            seqId = (Long)entry.getKey();
            break;
        }
        if (readOnly) {
            return lists[this.select(lists)];
        }
        this.cleanUpTrackFiles(seqId, seqId2TrackFiles);
        if (seqId < 0L) {
            this.initializeTrackFiles(System.currentTimeMillis());
            this.nextTrackFile = 0;
            return null;
        }
        this.initializeTrackFiles(Math.max(System.currentTimeMillis(), seqId + 1L));
        int winnerIndex = this.select(lists);
        this.nextTrackFile = 1 - winnerIndex;
        this.prevTimestamp = lists[winnerIndex].getTimestamp();
        return lists[winnerIndex];
    }

    void update(StoreFileTrackerProtos.StoreFileList.Builder builder) throws IOException {
        if (this.nextTrackFile < 0) {
            this.load(false);
        }
        long timestamp = Math.max(this.prevTimestamp + 1L, EnvironmentEdgeManager.currentTime());
        byte[] actualData = builder.setTimestamp(timestamp).build().toByteArray();
        CRC32 crc32 = new CRC32();
        crc32.update(actualData);
        int checksum = (int)crc32.getValue();
        FileSystem fs = this.ctx.getRegionFileSystem().getFileSystem();
        try (FSDataOutputStream out = fs.create(this.trackFiles[this.nextTrackFile], true);){
            out.writeInt(actualData.length);
            out.write(actualData);
            out.writeInt(checksum);
        }
        this.prevTimestamp = timestamp;
        this.nextTrackFile = 1 - this.nextTrackFile;
        try {
            fs.delete(this.trackFiles[this.nextTrackFile], false);
        }
        catch (IOException e) {
            LOG.debug("Failed to delete old track file {}, ignoring the exception", (Object)this.trackFiles[this.nextTrackFile], (Object)e);
        }
    }
}

