/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.mob;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Consumer;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.PrivateCellUtil;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.Tag;
import org.apache.hadoop.hbase.TagUtil;
import org.apache.hadoop.hbase.backup.HFileArchiver;
import org.apache.hadoop.hbase.client.ColumnFamilyDescriptor;
import org.apache.hadoop.hbase.client.RegionInfo;
import org.apache.hadoop.hbase.client.RegionInfoBuilder;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.hadoop.hbase.io.HFileLink;
import org.apache.hadoop.hbase.io.compress.Compression;
import org.apache.hadoop.hbase.io.crypto.Encryption;
import org.apache.hadoop.hbase.io.hfile.CacheConfig;
import org.apache.hadoop.hbase.io.hfile.HFile;
import org.apache.hadoop.hbase.io.hfile.HFileContext;
import org.apache.hadoop.hbase.io.hfile.HFileContextBuilder;
import org.apache.hadoop.hbase.mob.MobConstants;
import org.apache.hadoop.hbase.mob.MobFileName;
import org.apache.hadoop.hbase.regionserver.BloomType;
import org.apache.hadoop.hbase.regionserver.HStoreFile;
import org.apache.hadoop.hbase.regionserver.StoreFileWriter;
import org.apache.hadoop.hbase.regionserver.StoreUtils;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.ChecksumType;
import org.apache.hadoop.hbase.util.CommonFSUtils;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hbase.thirdparty.com.google.common.collect.ImmutableSetMultimap;
import org.apache.hbase.thirdparty.com.google.common.collect.SetMultimap;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public final class MobUtils {
    private static final Logger LOG = LoggerFactory.getLogger(MobUtils.class);
    public static final String SEP = "_";
    private static final ThreadLocal<SimpleDateFormat> LOCAL_FORMAT = new ThreadLocal<SimpleDateFormat>(){

        @Override
        protected SimpleDateFormat initialValue() {
            return new SimpleDateFormat("yyyyMMdd");
        }
    };

    private MobUtils() {
    }

    public static String formatDate(Date date) {
        return LOCAL_FORMAT.get().format(date);
    }

    public static Date parseDate(String dateString) throws ParseException {
        return LOCAL_FORMAT.get().parse(dateString);
    }

    public static boolean isMobReferenceCell(Cell cell) {
        Optional tag;
        return cell.getTagsLength() > 0 && (tag = PrivateCellUtil.getTag((Cell)cell, (byte)5)).isPresent();
    }

    private static Optional<Tag> getTableNameTag(Cell cell) {
        Optional tag = Optional.empty();
        if (cell.getTagsLength() > 0) {
            tag = PrivateCellUtil.getTag((Cell)cell, (byte)6);
        }
        return tag;
    }

    public static Optional<String> getTableNameString(Cell cell) {
        Optional<Tag> tag = MobUtils.getTableNameTag(cell);
        Optional<String> name = Optional.empty();
        if (tag.isPresent()) {
            name = Optional.of(Tag.getValueAsString((Tag)tag.get()));
        }
        return name;
    }

    public static Optional<TableName> getTableName(Cell cell) {
        Optional<Tag> maybe = MobUtils.getTableNameTag(cell);
        Optional<TableName> name = Optional.empty();
        if (maybe.isPresent()) {
            Tag tag = maybe.get();
            if (tag.hasArray()) {
                name = Optional.of(TableName.valueOf((byte[])tag.getValueArray(), (int)tag.getValueOffset(), (int)tag.getValueLength()));
            } else {
                ByteBuffer buffer = tag.getValueByteBuffer().duplicate();
                buffer.mark();
                buffer.position(tag.getValueOffset());
                buffer.limit(tag.getValueOffset() + tag.getValueLength());
                name = Optional.of(TableName.valueOf((ByteBuffer)buffer));
            }
        }
        return name;
    }

    public static boolean hasMobReferenceTag(List<Tag> tags) {
        if (!tags.isEmpty()) {
            for (Tag tag : tags) {
                if (tag.getType() != 5) continue;
                return true;
            }
        }
        return false;
    }

    public static boolean isRawMobScan(Scan scan) {
        byte[] raw = scan.getAttribute("hbase.mob.scan.raw");
        try {
            return raw != null && Bytes.toBoolean((byte[])raw);
        }
        catch (IllegalArgumentException e) {
            return false;
        }
    }

    public static boolean isRefOnlyScan(Scan scan) {
        byte[] refOnly = scan.getAttribute("hbase.mob.scan.ref.only");
        try {
            return refOnly != null && Bytes.toBoolean((byte[])refOnly);
        }
        catch (IllegalArgumentException e) {
            return false;
        }
    }

    public static boolean isCacheMobBlocks(Scan scan) {
        byte[] cache = scan.getAttribute("hbase.mob.cache.blocks");
        try {
            return cache != null && Bytes.toBoolean((byte[])cache);
        }
        catch (IllegalArgumentException e) {
            return false;
        }
    }

    public static void setCacheMobBlocks(Scan scan, boolean cacheBlocks) {
        scan.setAttribute("hbase.mob.cache.blocks", Bytes.toBytes((boolean)cacheBlocks));
    }

    public static void cleanExpiredMobFiles(FileSystem fs, Configuration conf, TableName tableName, ColumnFamilyDescriptor columnDescriptor, CacheConfig cacheConfig, long current) throws IOException {
        long timeToLive = columnDescriptor.getTimeToLive();
        if (Integer.MAX_VALUE == timeToLive) {
            return;
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeInMillis(current - timeToLive * 1000L);
        calendar.set(11, 0);
        calendar.set(12, 0);
        calendar.set(13, 0);
        Date expireDate = calendar.getTime();
        LOG.info("MOB HFiles older than " + expireDate.toGMTString() + " will be deleted!");
        FileStatus[] stats = null;
        Path mobTableDir = CommonFSUtils.getTableDir((Path)MobUtils.getMobHome(conf), (TableName)tableName);
        Path path = MobUtils.getMobFamilyPath(conf, tableName, columnDescriptor.getNameAsString());
        try {
            stats = fs.listStatus(path);
        }
        catch (FileNotFoundException e) {
            LOG.warn("Failed to find the mob file " + path, (Throwable)e);
        }
        if (null == stats) {
            return;
        }
        ArrayList<HStoreFile> filesToClean = new ArrayList<HStoreFile>();
        int deletedFileCount = 0;
        for (FileStatus file : stats) {
            String fileName = file.getPath().getName();
            try {
                if (HFileLink.isHFileLink(file.getPath())) {
                    HFileLink hfileLink = HFileLink.buildFromHFileLinkPattern(conf, file.getPath());
                    fileName = hfileLink.getOriginPath().getName();
                }
                Date fileDate = MobUtils.parseDate(MobFileName.getDateFromName(fileName));
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Checking file {}", (Object)fileName);
                }
                if (fileDate.getTime() >= expireDate.getTime()) continue;
                if (LOG.isDebugEnabled()) {
                    LOG.debug("{} is an expired file", (Object)fileName);
                }
                filesToClean.add(new HStoreFile(fs, file.getPath(), conf, cacheConfig, BloomType.NONE, true));
                if (filesToClean.size() < conf.getInt("hbase.master.mob.cleaner.batch.size.upper.bound", 10000)) continue;
                if (MobUtils.removeMobFiles(conf, fs, tableName, mobTableDir, columnDescriptor.getName(), filesToClean)) {
                    deletedFileCount += filesToClean.size();
                }
                filesToClean.clear();
            }
            catch (Exception e) {
                LOG.error("Cannot parse the fileName " + fileName, (Throwable)e);
            }
        }
        if (!filesToClean.isEmpty() && MobUtils.removeMobFiles(conf, fs, tableName, mobTableDir, columnDescriptor.getName(), filesToClean)) {
            deletedFileCount += filesToClean.size();
        }
        LOG.info("Table {} {} expired mob files in total are deleted", (Object)tableName, (Object)deletedFileCount);
    }

    public static Path getMobHome(Configuration conf) {
        Path hbaseDir = new Path(conf.get("hbase.rootdir"));
        return MobUtils.getMobHome(hbaseDir);
    }

    public static Path getMobHome(Path rootDir) {
        return new Path(rootDir, "mobdir");
    }

    public static Path getQualifiedMobRootDir(Configuration conf) throws IOException {
        Path hbaseDir = new Path(conf.get("hbase.rootdir"));
        Path mobRootDir = new Path(hbaseDir, "mobdir");
        FileSystem fs = mobRootDir.getFileSystem(conf);
        return mobRootDir.makeQualified(fs.getUri(), fs.getWorkingDirectory());
    }

    public static Path getMobTableDir(Path rootDir, TableName tableName) {
        return CommonFSUtils.getTableDir((Path)MobUtils.getMobHome(rootDir), (TableName)tableName);
    }

    public static Path getMobRegionPath(Configuration conf, TableName tableName) {
        return MobUtils.getMobRegionPath(new Path(conf.get("hbase.rootdir")), tableName);
    }

    public static Path getMobRegionPath(Path rootDir, TableName tableName) {
        Path tablePath = CommonFSUtils.getTableDir((Path)MobUtils.getMobHome(rootDir), (TableName)tableName);
        RegionInfo regionInfo = MobUtils.getMobRegionInfo(tableName);
        return new Path(tablePath, regionInfo.getEncodedName());
    }

    public static Path getMobFamilyPath(Configuration conf, TableName tableName, String familyName) {
        return new Path(MobUtils.getMobRegionPath(conf, tableName), familyName);
    }

    public static Path getMobFamilyPath(Path regionPath, String familyName) {
        return new Path(regionPath, familyName);
    }

    public static RegionInfo getMobRegionInfo(TableName tableName) {
        return RegionInfoBuilder.newBuilder((TableName)tableName).setStartKey(MobConstants.MOB_REGION_NAME_BYTES).setEndKey(HConstants.EMPTY_END_ROW).setSplit(false).setRegionId(0L).build();
    }

    public static boolean isMobRegionInfo(RegionInfo regionInfo) {
        return regionInfo == null ? false : MobUtils.getMobRegionInfo(regionInfo.getTable()).getEncodedName().equals(regionInfo.getEncodedName());
    }

    public static boolean isMobRegionName(TableName tableName, byte[] regionName) {
        return Bytes.equals((byte[])regionName, (byte[])MobUtils.getMobRegionInfo(tableName).getRegionName());
    }

    public static boolean removeMobFiles(Configuration conf, FileSystem fs, TableName tableName, Path tableDir, byte[] family, Collection<HStoreFile> storeFiles) {
        try {
            HFileArchiver.archiveStoreFiles(conf, fs, MobUtils.getMobRegionInfo(tableName), tableDir, family, storeFiles);
            LOG.info("Table {} {} expired mob files are deleted", (Object)tableName, (Object)storeFiles.size());
            return true;
        }
        catch (IOException e) {
            LOG.error("Failed to delete the mob files, table {}", (Object)tableName, (Object)e);
            return false;
        }
    }

    public static Cell createMobRefCell(Cell cell, byte[] fileName, Tag tableNameTag) {
        ArrayList<Tag> tags = new ArrayList<Tag>();
        tags.add(MobConstants.MOB_REF_TAG);
        tags.add(tableNameTag);
        return MobUtils.createMobRefCell(cell, fileName, TagUtil.fromList(tags));
    }

    public static Cell createMobRefCell(Cell cell, byte[] fileName, byte[] refCellTags) {
        byte[] refValue = Bytes.add((byte[])Bytes.toBytes((int)cell.getValueLength()), (byte[])fileName);
        return PrivateCellUtil.createCell((Cell)cell, (byte[])refValue, (byte[])TagUtil.concatTags((byte[])refCellTags, (Cell)cell));
    }

    public static StoreFileWriter createWriter(Configuration conf, FileSystem fs, ColumnFamilyDescriptor family, String date, Path basePath, long maxKeyCount, Compression.Algorithm compression, String startKey, CacheConfig cacheConfig, Encryption.Context cryptoContext, boolean isCompaction, String regionName) throws IOException {
        MobFileName mobFileName = MobFileName.create(startKey, date, UUID.randomUUID().toString().replaceAll("-", ""), regionName);
        return MobUtils.createWriter(conf, fs, family, mobFileName, basePath, maxKeyCount, compression, cacheConfig, cryptoContext, isCompaction);
    }

    public static StoreFileWriter createWriter(Configuration conf, FileSystem fs, ColumnFamilyDescriptor family, MobFileName mobFileName, Path basePath, long maxKeyCount, Compression.Algorithm compression, CacheConfig cacheConfig, Encryption.Context cryptoContext, boolean isCompaction) throws IOException {
        return MobUtils.createWriter(conf, fs, family, new Path(basePath, mobFileName.getFileName()), maxKeyCount, compression, cacheConfig, cryptoContext, StoreUtils.getChecksumType(conf), StoreUtils.getBytesPerChecksum(conf), family.getBlocksize(), BloomType.NONE, isCompaction);
    }

    public static StoreFileWriter createWriter(Configuration conf, FileSystem fs, ColumnFamilyDescriptor family, Path path, long maxKeyCount, Compression.Algorithm compression, CacheConfig cacheConfig, Encryption.Context cryptoContext, ChecksumType checksumType, int bytesPerChecksum, int blocksize, BloomType bloomType, boolean isCompaction) throws IOException {
        return MobUtils.createWriter(conf, fs, family, path, maxKeyCount, compression, cacheConfig, cryptoContext, checksumType, bytesPerChecksum, blocksize, bloomType, isCompaction, null);
    }

    public static StoreFileWriter createWriter(Configuration conf, FileSystem fs, ColumnFamilyDescriptor family, Path path, long maxKeyCount, Compression.Algorithm compression, CacheConfig cacheConfig, Encryption.Context cryptoContext, ChecksumType checksumType, int bytesPerChecksum, int blocksize, BloomType bloomType, boolean isCompaction, Consumer<Path> writerCreationTracker) throws IOException {
        CacheConfig writerCacheConf;
        if (compression == null) {
            compression = HFile.DEFAULT_COMPRESSION_ALGORITHM;
        }
        if (isCompaction) {
            writerCacheConf = new CacheConfig(cacheConfig);
            writerCacheConf.setCacheDataOnWrite(false);
        } else {
            writerCacheConf = cacheConfig;
        }
        HFileContext hFileContext = new HFileContextBuilder().withCompression(compression).withIncludesMvcc(true).withIncludesTags(true).withCompressTags(family.isCompressTags()).withChecksumType(checksumType).withBytesPerCheckSum(bytesPerChecksum).withBlockSize(blocksize).withHBaseCheckSum(true).withDataBlockEncoding(family.getDataBlockEncoding()).withEncryptionContext(cryptoContext).withCreateTime(EnvironmentEdgeManager.currentTime()).build();
        StoreFileWriter w = new StoreFileWriter.Builder(conf, writerCacheConf, fs).withFilePath(path).withBloomType(bloomType).withMaxKeyCount(maxKeyCount).withFileContext(hFileContext).withWriterCreationTracker(writerCreationTracker).build();
        return w;
    }

    public static boolean hasValidMobRefCellValue(Cell cell) {
        return cell.getValueLength() > 4;
    }

    public static int getMobValueLength(Cell cell) {
        return PrivateCellUtil.getValueAsInt((Cell)cell);
    }

    public static String getMobFileName(Cell cell) {
        return Bytes.toString((byte[])cell.getValueArray(), (int)(cell.getValueOffset() + 4), (int)(cell.getValueLength() - 4));
    }

    public static boolean hasMobColumns(TableDescriptor htd) {
        ColumnFamilyDescriptor[] hcds;
        for (ColumnFamilyDescriptor hcd : hcds = htd.getColumnFamilies()) {
            if (!hcd.isMobEnabled()) continue;
            return true;
        }
        return false;
    }

    public static List<ColumnFamilyDescriptor> getMobColumnFamilies(TableDescriptor htd) {
        ColumnFamilyDescriptor[] hcds;
        ArrayList<ColumnFamilyDescriptor> fams = new ArrayList<ColumnFamilyDescriptor>();
        for (ColumnFamilyDescriptor hcd : hcds = htd.getColumnFamilies()) {
            if (!hcd.isMobEnabled()) continue;
            fams.add(hcd);
        }
        return fams;
    }

    public static boolean isReadEmptyValueOnMobCellMiss(Scan scan) {
        byte[] readEmptyValueOnMobCellMiss = scan.getAttribute("empty.value.on.mobcell.miss");
        try {
            return readEmptyValueOnMobCellMiss != null && Bytes.toBoolean((byte[])readEmptyValueOnMobCellMiss);
        }
        catch (IllegalArgumentException e) {
            return false;
        }
    }

    public static boolean isMobFileExpired(ColumnFamilyDescriptor column, long current, String fileDate) {
        if (column.getMinVersions() > 0) {
            return false;
        }
        long timeToLive = column.getTimeToLive();
        if (Integer.MAX_VALUE == timeToLive) {
            return false;
        }
        Date expireDate = new Date(current - timeToLive * 1000L);
        expireDate = new Date(expireDate.getYear(), expireDate.getMonth(), expireDate.getDate());
        try {
            Date date = MobUtils.parseDate(fileDate);
            if (date.getTime() < expireDate.getTime()) {
                return true;
            }
        }
        catch (ParseException e) {
            LOG.warn("Failed to parse the date " + fileDate, (Throwable)e);
            return false;
        }
        return false;
    }

    public static byte[] serializeMobFileRefs(SetMultimap<TableName, String> mobRefSet) {
        if (mobRefSet != null && mobRefSet.size() > 0) {
            StringBuilder sb = new StringBuilder(100 + mobRefSet.size() * 105);
            boolean doubleSlash = false;
            for (TableName tableName : mobRefSet.keySet()) {
                if (doubleSlash) {
                    sb.append("//");
                } else {
                    doubleSlash = true;
                }
                sb.append(tableName).append("/");
                boolean comma = false;
                for (String refs : mobRefSet.get((Object)tableName)) {
                    if (comma) {
                        sb.append(",");
                    } else {
                        comma = true;
                    }
                    sb.append(refs);
                }
            }
            return Bytes.toBytes((String)sb.toString());
        }
        return HStoreFile.NULL_VALUE;
    }

    public static ImmutableSetMultimap.Builder<TableName, String> deserializeMobFileRefs(byte[] bytes) throws IllegalStateException {
        ImmutableSetMultimap.Builder map = ImmutableSetMultimap.builder();
        if (bytes.length > 1) {
            String[] tables;
            String s = Bytes.toString((byte[])bytes);
            for (String tableEnc : tables = s.split("//")) {
                int delim = tableEnc.indexOf(47);
                if (delim <= 0) {
                    throw new IllegalStateException("MOB reference data does not match expected encoding: no table name included before list of mob refs.");
                }
                TableName table = TableName.valueOf((String)tableEnc.substring(0, delim));
                Object[] refs = tableEnc.substring(delim + 1).split(",");
                map.putAll((Object)table, refs);
            }
        } else if (LOG.isDebugEnabled() && !Arrays.equals(HStoreFile.NULL_VALUE, bytes)) {
            LOG.debug("Serialized MOB file refs array was treated as the placeholder 'no entries' but didn't have the expected placeholder byte. expected={} and actual={}", (Object)Arrays.toString(HStoreFile.NULL_VALUE), (Object)Arrays.toString(bytes));
        }
        return map;
    }
}

