/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gravitino.iceberg.service.metrics;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableMap;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.time.Instant;
import java.util.Optional;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.apache.gravitino.iceberg.common.IcebergConfig;
import org.apache.gravitino.iceberg.service.IcebergRESTUtils;
import org.apache.gravitino.iceberg.service.metrics.DummyMetricsStore;
import org.apache.gravitino.iceberg.service.metrics.IcebergMetricsFormatter;
import org.apache.gravitino.iceberg.service.metrics.IcebergMetricsStore;
import org.apache.gravitino.iceberg.service.metrics.JDBCMetricsStore;
import org.apache.iceberg.catalog.Namespace;
import org.apache.iceberg.metrics.MetricsReport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IcebergMetricsManager {
    private static final Logger LOG = LoggerFactory.getLogger(IcebergMetricsManager.class);
    private static final ImmutableMap<String, String> ICEBERG_METRICS_STORE_NAMES = ImmutableMap.of((Object)"dummy", (Object)DummyMetricsStore.class.getCanonicalName(), (Object)"jdbc", (Object)JDBCMetricsStore.class.getCanonicalName());
    private final IcebergMetricsFormatter icebergMetricsFormatter;
    private final IcebergMetricsStore icebergMetricsStore;
    private final int retainDays;
    private BlockingQueue<MetricsReportWrapper> queue;
    private Thread metricsWriterThread;
    private volatile boolean isClosed = false;
    private Optional<ScheduledExecutorService> metricsCleanerExecutor = Optional.empty();

    public IcebergMetricsManager(IcebergConfig icebergConfig) {
        this.icebergMetricsFormatter = new IcebergMetricsFormatter();
        this.icebergMetricsStore = this.loadIcebergMetricsStore((String)icebergConfig.get(IcebergConfig.ICEBERG_METRICS_STORE));
        try {
            this.icebergMetricsStore.init(icebergConfig.getAllConfig());
        }
        catch (IOException e2) {
            LOG.warn("Iceberg metrics store init failed.", (Throwable)e2);
            throw new RuntimeException(e2);
        }
        this.retainDays = (Integer)icebergConfig.get(IcebergConfig.ICEBERG_METRICS_STORE_RETAIN_DAYS);
        if (this.retainDays > 0) {
            this.metricsCleanerExecutor = Optional.of(new ScheduledThreadPoolExecutor(1, new ThreadFactoryBuilder().setDaemon(true).setNameFormat("Iceberg-metrics-cleaner").setUncaughtExceptionHandler((t, e) -> LOG.error("Uncaught exception in thread {}.", (Object)t, (Object)e)).build()));
        }
        int queueCapacity = (Integer)icebergConfig.get(IcebergConfig.ICEBERG_METRICS_QUEUE_CAPACITY);
        this.queue = new LinkedBlockingQueue<MetricsReportWrapper>(queueCapacity);
        this.metricsWriterThread = new Thread(() -> this.writeMetrics());
        this.metricsWriterThread.setName("Iceberg-metrics-writer");
        this.metricsWriterThread.setDaemon(true);
    }

    public void start() {
        this.metricsWriterThread.start();
        this.metricsCleanerExecutor.ifPresent(executorService -> executorService.scheduleAtFixedRate(() -> {
            Instant now = Instant.now();
            Instant expireTime = IcebergRESTUtils.calculateNewTimestamp(now, -24 * this.retainDays);
            LOG.info("Try clean Iceberg expired metrics, {}.", (Object)expireTime);
            try {
                this.icebergMetricsStore.clean(expireTime);
            }
            catch (Exception e) {
                LOG.warn("Clean Iceberg metrics failed.", (Throwable)e);
            }
        }, 0L, 1L, TimeUnit.HOURS));
    }

    public boolean recordMetric(String catalogName, Namespace namespace, MetricsReport metricsReport) {
        if (this.isClosed) {
            this.logMetrics("Drop Iceberg metrics because Iceberg Metrics Manager is closed.", metricsReport);
            return false;
        }
        if (!this.queue.offer(new MetricsReportWrapper(catalogName, namespace, metricsReport))) {
            this.logMetrics("Drop Iceberg metrics because metrics queue is full.", metricsReport);
            return false;
        }
        return true;
    }

    public void close() {
        this.isClosed = true;
        this.metricsCleanerExecutor.ifPresent(executorService -> executorService.shutdownNow());
        if (this.icebergMetricsStore != null) {
            try {
                this.icebergMetricsStore.close();
            }
            catch (IOException e) {
                LOG.warn("Close Iceberg metrics store failed.", (Throwable)e);
            }
        }
        if (this.metricsWriterThread != null) {
            this.metricsWriterThread.interrupt();
            try {
                this.metricsWriterThread.join();
            }
            catch (InterruptedException e) {
                LOG.warn("Iceberg metrics manager is interrupted while join metrics writer thread.");
            }
        }
    }

    @VisibleForTesting
    IcebergMetricsStore getIcebergMetricsStore() {
        return this.icebergMetricsStore;
    }

    private void writeMetrics() {
        MetricsReportWrapper metricsReport;
        while (!Thread.currentThread().isInterrupted()) {
            try {
                metricsReport = this.queue.take();
            }
            catch (InterruptedException e) {
                LOG.warn("Iceberg Metrics writer thread is interrupted.");
                break;
            }
            this.doRecordMetric(metricsReport.getCatalog(), metricsReport.getNamespace(), metricsReport.getMetricsReport());
        }
        metricsReport = (MetricsReportWrapper)this.queue.poll();
        while (metricsReport != null) {
            this.logMetrics("Drop Iceberg metrics because it's time to close metrics store.", metricsReport.getMetricsReport());
            metricsReport = (MetricsReportWrapper)this.queue.poll();
        }
    }

    private IcebergMetricsStore loadIcebergMetricsStore(String metricsStoreName) {
        if (metricsStoreName == null) {
            metricsStoreName = "dummy";
        }
        String metricsStoreClass = (String)ICEBERG_METRICS_STORE_NAMES.getOrDefault((Object)metricsStoreName, (Object)metricsStoreName);
        LOG.info("Load Iceberg metrics store: {}.", (Object)metricsStoreClass);
        try {
            return (IcebergMetricsStore)Class.forName(metricsStoreClass).getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
        }
        catch (Exception e) {
            LOG.error("Failed to create and initialize Iceberg metrics store by name {}.", (Object)metricsStoreName, (Object)e);
            throw new RuntimeException(e);
        }
    }

    private void logMetrics(String message, MetricsReport metricsReport) {
        LOG.info("{} {}.", (Object)message, (Object)this.icebergMetricsFormatter.toPrintableString(metricsReport));
    }

    private void doRecordMetric(String catalog, Namespace namespace, MetricsReport metricsReport) {
        try {
            this.icebergMetricsStore.recordMetric(catalog, namespace, metricsReport);
        }
        catch (Exception e) {
            LOG.warn("Write Iceberg metrics failed.", (Throwable)e);
        }
    }

    private static class MetricsReportWrapper {
        private final String catalog;
        private final Namespace namespace;
        private final MetricsReport metricsReport;

        public MetricsReportWrapper(String catalog, Namespace namespace, MetricsReport metricsReport) {
            this.catalog = catalog;
            this.namespace = namespace;
            this.metricsReport = metricsReport;
        }

        public Namespace getNamespace() {
            return this.namespace;
        }

        public MetricsReport getMetricsReport() {
            return this.metricsReport;
        }

        public String getCatalog() {
            return this.catalog;
        }
    }
}

