/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.server.compaction;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.PriorityQueue;
import java.util.Set;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.druid.java.util.common.DateTimes;
import org.apache.druid.java.util.common.Intervals;
import org.apache.druid.java.util.common.JodaUtils;
import org.apache.druid.java.util.common.granularity.Granularity;
import org.apache.druid.java.util.common.guava.Comparators;
import org.apache.druid.java.util.common.logger.Logger;
import org.apache.druid.server.compaction.CompactionCandidate;
import org.apache.druid.server.compaction.CompactionCandidateSearchPolicy;
import org.apache.druid.server.compaction.CompactionSegmentIterator;
import org.apache.druid.server.compaction.CompactionStatus;
import org.apache.druid.server.compaction.CompactionStatusTracker;
import org.apache.druid.server.coordinator.DataSourceCompactionConfig;
import org.apache.druid.timeline.DataSegment;
import org.apache.druid.timeline.Partitions;
import org.apache.druid.timeline.SegmentTimeline;
import org.apache.druid.timeline.TimelineObjectHolder;
import org.apache.druid.timeline.partition.NumberedPartitionChunk;
import org.apache.druid.timeline.partition.NumberedShardSpec;
import org.apache.druid.timeline.partition.PartitionChunk;
import org.apache.druid.timeline.partition.ShardSpec;
import org.apache.druid.utils.CollectionUtils;
import org.apache.druid.utils.Streams;
import org.joda.time.DateTime;
import org.joda.time.Interval;
import org.joda.time.Period;
import org.joda.time.ReadableInstant;
import org.joda.time.ReadableInterval;
import org.joda.time.ReadablePeriod;

public class DataSourceCompactibleSegmentIterator
implements CompactionSegmentIterator {
    private static final Logger log = new Logger(DataSourceCompactibleSegmentIterator.class);
    private final String dataSource;
    private final DataSourceCompactionConfig config;
    private final CompactionStatusTracker statusTracker;
    private final CompactionCandidateSearchPolicy searchPolicy;
    private final List<CompactionCandidate> compactedSegments = new ArrayList<CompactionCandidate>();
    private final List<CompactionCandidate> skippedSegments = new ArrayList<CompactionCandidate>();
    private final Set<Interval> queuedIntervals = new HashSet<Interval>();
    private final PriorityQueue<CompactionCandidate> queue;

    public DataSourceCompactibleSegmentIterator(DataSourceCompactionConfig config, SegmentTimeline timeline, List<Interval> skipIntervals, CompactionCandidateSearchPolicy searchPolicy, CompactionStatusTracker statusTracker) {
        this.statusTracker = statusTracker;
        this.config = config;
        this.dataSource = config.getDataSource();
        this.searchPolicy = searchPolicy;
        this.queue = new PriorityQueue<CompactionCandidate>(searchPolicy);
        this.populateQueue(timeline, skipIntervals);
    }

    private void populateQueue(SegmentTimeline timeline, List<Interval> skipIntervals) {
        if (timeline != null && !timeline.isEmpty()) {
            List<Interval> searchIntervals;
            SegmentTimeline originalTimeline = null;
            if (this.config.getSegmentGranularity() != null) {
                Set segments = timeline.findNonOvershadowedObjectsInInterval(Intervals.ETERNITY, Partitions.ONLY_COMPLETE);
                ArrayList<DataSegment> partialEternitySegments = new ArrayList<DataSegment>();
                for (DataSegment segment : segments) {
                    if (!Intervals.ETERNITY.getStart().equals((Object)segment.getInterval().getStart()) && !Intervals.ETERNITY.getEnd().equals((Object)segment.getInterval().getEnd())) continue;
                    partialEternitySegments.add(segment);
                }
                if (!partialEternitySegments.isEmpty()) {
                    CompactionCandidate candidatesWithStatus = CompactionCandidate.from(partialEternitySegments).withCurrentStatus(CompactionStatus.skipped("Segments have partial-eternity intervals", new Object[0]));
                    this.skippedSegments.add(candidatesWithStatus);
                    this.statusTracker.onCompactionStatusComputed(candidatesWithStatus, this.config);
                    return;
                }
                SegmentTimeline timelineWithConfiguredSegmentGranularity = new SegmentTimeline();
                HashMap<Interval, Set> intervalToPartitionMap = new HashMap<Interval, Set>();
                for (DataSegment segment : segments) {
                    for (Interval interval : this.config.getSegmentGranularity().getIterable(segment.getInterval())) {
                        intervalToPartitionMap.computeIfAbsent(interval, k -> new HashSet()).add(segment);
                    }
                }
                String temporaryVersion = DateTimes.nowUtc().toString();
                for (Map.Entry partitionsPerInterval : intervalToPartitionMap.entrySet()) {
                    Interval interval;
                    interval = (Interval)partitionsPerInterval.getKey();
                    int partitionNum = 0;
                    Set segmentSet = (Set)partitionsPerInterval.getValue();
                    int partitions = segmentSet.size();
                    for (DataSegment segment : segmentSet) {
                        DataSegment segmentsForCompact = segment.withShardSpec((ShardSpec)new NumberedShardSpec(partitionNum, partitions));
                        timelineWithConfiguredSegmentGranularity.add(interval, (Object)temporaryVersion, (PartitionChunk)NumberedPartitionChunk.make((int)partitionNum, (int)partitions, (Object)segmentsForCompact));
                        ++partitionNum;
                    }
                }
                originalTimeline = timeline;
                timeline = timelineWithConfiguredSegmentGranularity;
            }
            if (!(searchIntervals = this.findInitialSearchInterval(timeline, skipIntervals)).isEmpty()) {
                this.findAndEnqueueSegmentsToCompact(new CompactibleSegmentIterator(timeline, searchIntervals, originalTimeline));
            } else {
                log.warn("Skipping compaction for datasource[%s] as it has no compactible segments.", new Object[]{this.dataSource});
            }
        }
    }

    @Override
    public List<CompactionCandidate> getCompactedSegments() {
        return this.compactedSegments;
    }

    @Override
    public List<CompactionCandidate> getSkippedSegments() {
        return this.skippedSegments;
    }

    @Override
    public boolean hasNext() {
        return !this.queue.isEmpty();
    }

    @Override
    public CompactionCandidate next() {
        if (this.hasNext()) {
            return this.queue.poll();
        }
        throw new NoSuchElementException();
    }

    private void findAndEnqueueSegmentsToCompact(CompactibleSegmentIterator compactibleSegmentIterator) {
        while (compactibleSegmentIterator.hasNext()) {
            Object segments = compactibleSegmentIterator.next();
            if (CollectionUtils.isNullOrEmpty((Collection)segments) || segments.size() == 1 && ((DataSegment)segments.get(0)).isTombstone()) continue;
            CompactionCandidate candidates = CompactionCandidate.from((List<DataSegment>)segments);
            CompactionStatus compactionStatus = this.statusTracker.computeCompactionStatus(candidates, this.config, this.searchPolicy);
            CompactionCandidate candidatesWithStatus = candidates.withCurrentStatus(compactionStatus);
            this.statusTracker.onCompactionStatusComputed(candidatesWithStatus, this.config);
            if (compactionStatus.isComplete()) {
                this.compactedSegments.add(candidatesWithStatus);
                continue;
            }
            if (compactionStatus.isSkipped()) {
                this.skippedSegments.add(candidatesWithStatus);
                continue;
            }
            if (this.queuedIntervals.contains(candidates.getUmbrellaInterval())) continue;
            this.queue.add(candidatesWithStatus);
            this.queuedIntervals.add(candidates.getUmbrellaInterval());
        }
    }

    private List<Interval> findInitialSearchInterval(SegmentTimeline timeline, @Nullable List<Interval> skipIntervals) {
        Period skipOffset = this.config.getSkipOffsetFromLatest();
        Preconditions.checkArgument((timeline != null && !timeline.isEmpty() ? 1 : 0) != 0, (Object)"timeline should not be null or empty");
        Preconditions.checkNotNull((Object)skipOffset, (Object)"skipOffset");
        TimelineObjectHolder first = (TimelineObjectHolder)Preconditions.checkNotNull((Object)timeline.first(), (Object)"first");
        TimelineObjectHolder last = (TimelineObjectHolder)Preconditions.checkNotNull((Object)timeline.last(), (Object)"last");
        Interval latestSkipInterval = DataSourceCompactibleSegmentIterator.computeLatestSkipInterval(this.config.getSegmentGranularity(), last.getInterval().getEnd(), skipOffset);
        List<Interval> allSkipIntervals = DataSourceCompactibleSegmentIterator.sortAndAddSkipIntervalFromLatest(latestSkipInterval, skipIntervals);
        for (Interval skipInterval : allSkipIntervals) {
            ArrayList<DataSegment> segments = new ArrayList<DataSegment>(timeline.findNonOvershadowedObjectsInInterval(skipInterval, Partitions.ONLY_COMPLETE));
            if (CollectionUtils.isNullOrEmpty(segments)) continue;
            CompactionCandidate candidates = CompactionCandidate.from(segments);
            CompactionStatus reason = candidates.getUmbrellaInterval().overlaps((ReadableInterval)latestSkipInterval) ? CompactionStatus.skipped("skip offset from latest[%s]", skipOffset) : CompactionStatus.skipped("interval locked by another task", new Object[0]);
            CompactionCandidate candidatesWithStatus = candidates.withCurrentStatus(reason);
            this.skippedSegments.add(candidatesWithStatus);
            this.statusTracker.onCompactionStatusComputed(candidatesWithStatus, this.config);
        }
        Interval totalInterval = new Interval((ReadableInstant)first.getInterval().getStart(), (ReadableInstant)last.getInterval().getEnd());
        List<Interval> filteredInterval = DataSourceCompactibleSegmentIterator.filterSkipIntervals(totalInterval, allSkipIntervals);
        ArrayList<Interval> searchIntervals = new ArrayList<Interval>();
        for (Interval lookupInterval : filteredInterval) {
            if (Intervals.ETERNITY.getStart().equals((Object)lookupInterval.getStart()) || Intervals.ETERNITY.getEnd().equals((Object)lookupInterval.getEnd())) {
                log.warn("Cannot compact datasource[%s] since interval[%s] coincides with ETERNITY.", new Object[]{this.dataSource, lookupInterval});
                return Collections.emptyList();
            }
            List segments = timeline.findNonOvershadowedObjectsInInterval(lookupInterval, Partitions.ONLY_COMPLETE).stream().filter(segment -> lookupInterval.contains((ReadableInterval)segment.getInterval())).collect(Collectors.toList());
            if (segments.isEmpty()) continue;
            DateTime searchStart = segments.stream().map(segment -> segment.getId().getIntervalStart()).min(Comparator.naturalOrder()).orElseThrow(AssertionError::new);
            DateTime searchEnd = segments.stream().map(segment -> segment.getId().getIntervalEnd()).max(Comparator.naturalOrder()).orElseThrow(AssertionError::new);
            searchIntervals.add(new Interval((ReadableInstant)searchStart, (ReadableInstant)searchEnd));
        }
        return searchIntervals;
    }

    static Interval computeLatestSkipInterval(@Nullable Granularity configuredSegmentGranularity, DateTime latestDataTimestamp, Period skipOffsetFromLatest) {
        if (configuredSegmentGranularity == null) {
            return new Interval((ReadablePeriod)skipOffsetFromLatest, (ReadableInstant)latestDataTimestamp);
        }
        DateTime skipFromLastest = new DateTime((Object)latestDataTimestamp, latestDataTimestamp.getZone()).minus((ReadablePeriod)skipOffsetFromLatest);
        DateTime skipOffsetBucketToSegmentGranularity = configuredSegmentGranularity.bucketStart(skipFromLastest);
        return new Interval((ReadableInstant)skipOffsetBucketToSegmentGranularity, (ReadableInstant)latestDataTimestamp);
    }

    @VisibleForTesting
    static List<Interval> sortAndAddSkipIntervalFromLatest(Interval skipFromLatest, @Nullable List<Interval> skipIntervals) {
        ArrayList<Interval> nonNullSkipIntervals;
        ArrayList<Object> arrayList = nonNullSkipIntervals = skipIntervals == null ? new ArrayList<Interval>(1) : new ArrayList(skipIntervals.size());
        if (skipIntervals != null) {
            ArrayList<Interval> sortedSkipIntervals = new ArrayList<Interval>(skipIntervals);
            sortedSkipIntervals.sort(Comparators.intervalsByStartThenEnd());
            ArrayList<Interval> overlapIntervals = new ArrayList<Interval>();
            for (Interval interval : sortedSkipIntervals) {
                if (interval.overlaps((ReadableInterval)skipFromLatest)) {
                    overlapIntervals.add(interval);
                    continue;
                }
                nonNullSkipIntervals.add(interval);
            }
            if (!overlapIntervals.isEmpty()) {
                overlapIntervals.add(skipFromLatest);
                nonNullSkipIntervals.add(JodaUtils.umbrellaInterval(overlapIntervals));
            } else {
                nonNullSkipIntervals.add(skipFromLatest);
            }
        } else {
            nonNullSkipIntervals.add(skipFromLatest);
        }
        return nonNullSkipIntervals;
    }

    @VisibleForTesting
    static List<Interval> filterSkipIntervals(Interval totalInterval, List<Interval> skipIntervals) {
        ArrayList<Interval> filteredIntervals = new ArrayList<Interval>(skipIntervals.size() + 1);
        DateTime remainingStart = totalInterval.getStart();
        DateTime remainingEnd = totalInterval.getEnd();
        for (Interval skipInterval : skipIntervals) {
            if (skipInterval.getStart().isBefore((ReadableInstant)remainingStart) && skipInterval.getEnd().isAfter((ReadableInstant)remainingStart)) {
                remainingStart = skipInterval.getEnd();
                continue;
            }
            if (skipInterval.getStart().isBefore((ReadableInstant)remainingEnd) && skipInterval.getEnd().isAfter((ReadableInstant)remainingEnd)) {
                remainingEnd = skipInterval.getStart();
                continue;
            }
            if (!remainingStart.isAfter((ReadableInstant)skipInterval.getStart()) && !remainingEnd.isBefore((ReadableInstant)skipInterval.getEnd())) {
                filteredIntervals.add(new Interval((ReadableInstant)remainingStart, (ReadableInstant)skipInterval.getStart()));
                remainingStart = skipInterval.getEnd();
                continue;
            }
            log.debug("skipInterval[%s] is not contained in remainingInterval[%s]", new Object[]{skipInterval, new Interval((ReadableInstant)remainingStart, (ReadableInstant)remainingEnd)});
        }
        if (!remainingStart.equals((Object)remainingEnd)) {
            filteredIntervals.add(new Interval((ReadableInstant)remainingStart, (ReadableInstant)remainingEnd));
        }
        return filteredIntervals;
    }

    private static class CompactibleSegmentIterator
    implements Iterator<List<DataSegment>> {
        private final List<TimelineObjectHolder<String, DataSegment>> holders;
        @Nullable
        private final SegmentTimeline originalTimeline;

        CompactibleSegmentIterator(SegmentTimeline timeline, List<Interval> totalIntervalsToSearch, @Nullable SegmentTimeline originalTimeline) {
            this.holders = totalIntervalsToSearch.stream().flatMap(interval -> timeline.lookup(interval).stream().filter(holder -> this.isCompactibleHolder((Interval)interval, (TimelineObjectHolder<String, DataSegment>)holder))).collect(Collectors.toList());
            this.originalTimeline = originalTimeline;
        }

        private boolean isCompactibleHolder(Interval searchInterval, TimelineObjectHolder<String, DataSegment> holder) {
            long partitionBytes;
            Iterator chunks = holder.getObject().iterator();
            if (!chunks.hasNext()) {
                return false;
            }
            PartitionChunk firstChunk = (PartitionChunk)chunks.next();
            if (!searchInterval.contains((ReadableInterval)((DataSegment)firstChunk.getObject()).getInterval())) {
                return false;
            }
            for (partitionBytes = ((DataSegment)firstChunk.getObject()).getSize(); partitionBytes == 0L && chunks.hasNext(); partitionBytes += ((DataSegment)((PartitionChunk)chunks.next()).getObject()).getSize()) {
            }
            return partitionBytes > 0L;
        }

        @Override
        public boolean hasNext() {
            return !this.holders.isEmpty();
        }

        @Override
        public List<DataSegment> next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            TimelineObjectHolder<String, DataSegment> timelineObjectHolder = this.holders.remove(this.holders.size() - 1);
            List<DataSegment> candidates = Streams.sequentialStreamFrom((Iterable)timelineObjectHolder.getObject()).map(PartitionChunk::getObject).collect(Collectors.toList());
            if (this.originalTimeline == null) {
                return candidates;
            }
            Interval umbrellaInterval = JodaUtils.umbrellaInterval((Iterable)candidates.stream().map(DataSegment::getInterval).collect(Collectors.toList()));
            return Lists.newArrayList((Iterable)this.originalTimeline.findNonOvershadowedObjectsInInterval(umbrellaInterval, Partitions.ONLY_COMPLETE));
        }
    }
}

