// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

#include "arrow/flight/sql/odbc/odbc_impl/accessors/decimal_array_accessor.h"

#include "arrow/array.h"
#include "arrow/scalar.h"

namespace arrow::flight::sql::odbc {

template <typename ARROW_ARRAY, CDataType TARGET_TYPE>
DecimalArrayFlightSqlAccessor<ARROW_ARRAY, TARGET_TYPE>::DecimalArrayFlightSqlAccessor(
    Array* array)
    : FlightSqlAccessor<ARROW_ARRAY, TARGET_TYPE,
                        DecimalArrayFlightSqlAccessor<ARROW_ARRAY, TARGET_TYPE>>(array),
      data_type_(static_cast<Decimal128Type*>(array->type().get())) {}

template <>
RowStatus
DecimalArrayFlightSqlAccessor<Decimal128Array, CDataType_NUMERIC>::MoveSingleCellImpl(
    ColumnBinding* binding, int64_t arrow_row, int64_t i, int64_t& value_offset,
    bool update_value_offset, Diagnostics& diagnostics) {
  auto result = &(static_cast<NUMERIC_STRUCT*>(binding->buffer)[i]);
  int32_t original_scale = data_type_->scale();

  const uint8_t* bytes = this->GetArray()->Value(arrow_row);
  Decimal128 value(bytes);
  if (original_scale != binding->scale) {
    const Status& status = value.Rescale(original_scale, binding->scale).Value(&value);
    util::ThrowIfNotOK(status);
  }
  if (!value.FitsInPrecision(binding->precision)) {
    throw DriverException("Decimal value doesn't fit in precision " +
                          std::to_string(binding->precision));
  }

  result->sign = value.IsNegative() ? 0 : 1;

  // Take the absolute value since the ODBC SQL_NUMERIC_STRUCT holds
  // a positive-only number.
  if (value.IsNegative()) {
    Decimal128 abs_value = Decimal128::Abs(value);
    abs_value.ToBytes(result->val);
  } else {
    value.ToBytes(result->val);
  }
  result->precision = static_cast<uint8_t>(binding->precision);
  result->scale = static_cast<int8_t>(binding->scale);

  result->precision = data_type_->precision();

  if (binding->str_len_buffer) {
    binding->str_len_buffer[i] = static_cast<ssize_t>(GetCellLengthImpl(binding));
  }

  return RowStatus_SUCCESS;
}

template <typename ARROW_ARRAY, CDataType TARGET_TYPE>
size_t DecimalArrayFlightSqlAccessor<ARROW_ARRAY, TARGET_TYPE>::GetCellLengthImpl(
    ColumnBinding* binding) const {
  return sizeof(NUMERIC_STRUCT);
}

template class DecimalArrayFlightSqlAccessor<Decimal128Array, CDataType_NUMERIC>;

}  // namespace arrow::flight::sql::odbc
